;========================================================================
; Toddy, By Eric Tauck
;
; Toddy is a DOS command line editor TSR.  Some major features include
; enhanced editing commands, command history, file name completion, and
; macros.
;
; This source code is written for WASM and requires the WASM library
; files. Make sure that the all INCLUDE statements have the correct path
; to the library files.  Note that there is more than one block of
; INCLUDE statements.
;
;------------------------------------------------------------------------
; Heap Layout:
;
; The history (command stack) grows upward and the macro table grows
; downward.  The search data for the file name completion command is
; saved in the heap free space.
;
; HIGHMEM  macend
;    |       :
;    |       :
;    |     macbeg
;    |
;    |     (free)
;    |
;    |     hisend
;    |       :
;    |       :
;  LOWMEM  hisbeg
;
;------------------------------------------------------------------------
; Macro Interpreter:
;
; 1. User input is saved to start of the buffer:
;
;    [<input>-----------------------]
;    |                              |
;    bufbeg                         bufend
;
; 2. When the command is executed, the user input is copied to the end of
;    the buffer:
;
;    [----------------------<input>]
;
; 3. The first field (the command) is copied to the start of the buffer:
;
;    [<command>---------<arguments>]
;
; 4. If the command is a macro, the macro is expanded to the start of
;    the buffer using the arguments at the end of the buffer:
;
;    [<expansion>-------<arguments>]
;
; 5. If a macro was expanded, go back to step 2 (the macro expansion
;    becomes the input).  Steps 2 - 4 are are repeated for every nested
;    macro.
;
; 6. When all macros have been expanded (the command doesn't match any
;    macros), the input is copied back to the start of the buffer:
;
;    [<input>-----------------------]
;
; 7. Finally, the contents of the input buffer is copied to the address
;    passed to INT 21 function 0A.
;
; Macro interpeter routine hirearchy:
;
;  1. The topmost interpret code is in the interpret section of input
;     (starting around the label inter1).  This code calls Src_Read and
;     Src_Back to get and unget characters.
;
;  2. The routines Src_Read and Src_Back reads characters from the macro
;     text, replacing most of the special characters sequences (like $T,
;     $1, etc).  Src_Read calls itself to parse macro arguments, and
;     calls Src_Back when it has read too far.  Src_Read does NOT process
;     macro variables, like $(x).  Src_Read calls Src_Load and Src_Unload
;     to get and unget raw characters.
;
;  3. The routines Src_Load and Src_Unload exist only to process macro
;     variables.  When Src_Read sees the start of a macro variable, it
;     calls Src_Var, which pushes the current source location and
;     switches to the variable.  When Src_Load finds the end of a
;     variable, it pops the source location.
;
; Src_Back and Src_Load will "unpop" macro variable levels.  This assumes
; the popped variable table entries are still valid, since the unget
; information only consists of pointers to the variable stack -- the
; actual table data is not saved. A macro interpret error will occur if
; the following ever happens:
;
;   push source   ;var1
;   push source   ;var2
;     save unget
;   pop source    ;var2
;   pop source    ;var1
;   push source   ;var3, corrupts popped var1 data
;   unget         ;goes back to pushed var1/var2
;
; Though this could theoretically happen, I couldn't think of any
; macros that actually cause this to happen.

        JUMP+
        UNUSED+

VER_HI  EQU     6                       ;high version number (0 - 9)
VER_LO  EQU     15                      ;low version number (00 - 99)

MPLX1   EQU     0D7H                    ;multiplex number (C0 - FF)
MPLX2   EQU     0C2H                    ;multiplex subfunction (00 - FF)

DELIM   EQU     ' '                     ;macro argument delimiter
SWITCH  EQU     '/'                     ;switch character
COMMENT EQU     ';'                     ;comment character
COMMAND EQU     13                      ;command delimiter
DCURSOR EQU     0FFFFH                  ;default cursor setting

MAXINP  EQU     255                     ;maximum input size (min input buf)
MAXNAM  EQU     60                      ;maximum file name pattern (w/o NUL)
MAXLIN  EQU     512                     ;maximum read/write line length
MAXARG  EQU     65                      ;maximum arguments per line (approx)
MAXLOO  EQU     4000                    ;maximum key loops (approx)
SEAREC  EQU     43                      ;bytes for DOS file search record
XHEAP   EQU     MAXNAM + 1 + SEAREC + 1 ;extra heap needed for name matching

RES_OFF EQU     0                       ;relocation offset ;-- used by
RES_ID  EQU     'TO'                    ;resident ID       ;      MULPLX?.ASM

;--- flags (for variable 'flags')

INSERT  EQU     01H             ;insert mode
IRESET  EQU     02H             ;reset insert mode each entry
IVALUE  EQU     04H             ;insert setting if reset
CASE    EQU     08H             ;case sensitivity
CURSOR  EQU     10H             ;change cursor scan pattern
ENABLE  EQU     20H             ;input enabled
SECINS  EQU     40H             ;secondary insert flag
BREAK   EQU     80H             ;break pressed (must be high bit)

;--- more flags (for variable 'flags2')

HTRACE  EQU     01H             ;history trace recover
FERROR  EQU     02H             ;name search failed or pattern too long
PURHIS  EQU     04H             ;pure history
DELHIS  EQU     08H             ;delete selected history entry
REDISP  EQU     10H             ;DON'T display line (clear if chained cmd)
FBEGUN  EQU     20H             ;name search started (matches redisplay bit)
HRESET  EQU     40H             ;history reset (matches redisplay bit)
UNIQUE  EQU     80H             ;save unique lines only

;--- interpret flags (for variable 'flags3')

SINK    EQU     01H             ;sink parameter
QUOTE   EQU     02H             ;remove removable quotes
PARA1   EQU     04H             ;inside all argument insertion
PARA2   EQU     08H             ;inside single argument insertion
SRCEOL  EQU     10H             ;inside tail insertion
RECURS  EQU     20H             ;calling Src_Read recursively
SUBST   EQU     40H             ;substitute character
RUNCMD  EQU     80H             ;run command

;--- oops, need some more flags (for variable 'flags4')

DOSV32  EQU     01H             ;DOS version 3.2 or higher
SLASHED EQU     02H             ;marked directory names
PRIVATE EQU     04H             ;private data under Windows
HLOCK   EQU     08H             ;history lock
HSKIP   EQU     10H             ;skip history save
ASINK   EQU     20H             ;automatic sink

;--- redisplay and other flags

UPDLIN  EQU     01H             ;update whole line
UPDRIG  EQU     02H             ;update right of cursor
UPDOLD  EQU     04H             ;update last location
UPDCUR  EQU     08H             ;update cursor location (set in Redraw)
CLRCUR  EQU     10H             ;clear cursor block
;FBEGUN EQU     20H             ;this flag used here and in 'flags2'
;HRESET EQU     40H             ;this flag used here and in 'flags2'
RESET   EQU     80H             ;reset search

;--- installation configuration flags (for 'flags5')

QUIET   EQU     01H             ;install in quiet mode

;--- installation flags (for 'iflags')

PREAD   EQU     01H             ;read at least one switch
DEFCFG  EQU     02H             ;loading default configuration
LODCFG  EQU     04H             ;within Load_Config

;--- macro definition type

DEFANY  EQU     0               ;macro or key assignment
DEFMAC  EQU     1               ;macro only
DEFKEY  EQU     2               ;key assignment only

        jmps    stamp1          ;goto stamp code (avoid 07H in the offset)

;------------------------------------------------------------------------
; Save the date / time stamp.

        INCLUDE 'c:\src\wasm\stamp'

        DB      13,'  ',13,10
        DB      'Toddy '
        DB      VER_HI+'0', VER_LO / 10 + '0', VER_LO \ 10 + '0'
        DB      '-'

        TimSta

        DB      ' Eric Tauck [warp@earthling.net]',13,10
        DB      26

stamp1  jmp     Install

;************************************************************************
; Relocated Resident Data
;
; Note: data begins in PSP starting at offset 5CH.

FIXORG  EQU     $
        ORG     5CH

REL_TAR

;=== initialized data

flags   LABEL   BYTE            ;flags
        ORG     +1
        ORG     +1              ;zero byte attached to flags2, see NOTE_A
flags2  LABEL   BYTE            ;more flags
        ORG     +1
flags4  LABEL   BYTE            ;more flags
        ORG     +1
minlen  LABEL   BYTE            ;minimum input length
        ORG     +1
liter   LABEL   BYTE            ;literal
        ORG     +1
quote1  LABEL   BYTE            ;quote
        ORG     +1
quote2  LABEL   BYTE            ;quote
        ORG     +1
allarg  LABEL   BYTE            ;argument continuation
        ORG     +1
atext   LABEL   BYTE            ;text attribute
        ORG     +1
aend    LABEL   BYTE            ;end attribute
        ORG     +1
acurs   LABEL   BYTE            ;cursor attribute
        ORG     +1
cins    LABEL   WORD            ;cursor insert pattern
        ORG     +2
cover   LABEL   WORD            ;cursor overwrite pattern
        ORG     +2
cexit   LABEL   WORD            ;cursor exit pattern
        ORG     +2
sizchk  LABEL   BYTE            ;buffer size check
        ORG     +1
namsiz  LABEL   WORD            ;size of name save area
        ORG     +2
namtyp  LABEL   WORD            ;attribute for name matching
        ORG     +2
namcas  LABEL   WORD            ;case conversion for name matching
        ORG     +2
keyfun  LABEL   BYTE            ;key input function
        ORG     +1

        ORG     +2              ;input buffer size ;
        ORG     +2              ;key buffer size   ;- must be resident for /A
        ORG     +2              ;macro nest        ; ;
varmax  LABEL   BYTE            ;variable nest       ;
        ORG     +1                                   ;
        ORG     +2              ;variable length   ;-/

        ORG     +2              ;history size
        ORG     +2              ;macro size

subtab  LABEL   BYTE            ;start of substitution table
param   LABEL   BYTE            ;parameter
        ORG     +2
chain   LABEL   BYTE            ;chain
        ORG     +2
remove  LABEL   BYTE            ;removable quote
        ORG     +2
        ORG     +2
        ORG     +2
        ORG     +2
subend  LABEL   BYTE            ;end of substitution table

varchr1 LABEL   BYTE            ;start of variable
        ORG     +1
varchr2 LABEL   BYTE            ;end of variable
        ORG     +1

        ORG     +7              ;select window attributes

curhis  LABEL   WORD            ;current history entry
        ORG     +2
cdef    LABEL   WORD            ;default cursor
        ORG     +2
loaded  LABEL   WORD            ;preloaded input length
        ORG     +2

;--- heap addresses

hisbeg  LABEL   WORD            ;beginning of history
        ORG     +2
hisend  LABEL   WORD            ;end of history
        ORG     +2
macbeg  LABEL   WORD            ;beginning of macros
        ORG     +2
macend  LABEL   WORD            ;end of macros
        ORG     +2

;=== end of initialized data

flags3  LABEL   WORD            ;interpret flags
        ORG     +1              ;  low byte is flags
        ORG     +1              ;  high byte is varible nest level
oldloc  LABEL   WORD            ;old location
        ORG     +2
histmp  LABEL   WORD            ;temporary history variable
        ORG     +2
search  LABEL   BYTE            ;number search bytes, high byte always zero
        ORG     +2
rfirst  LABEL   WORD            ;first reverse search entry found
        ORG     +2
ffirst  LABEL   WORD            ;first forward search entry found
        ORG     +2
lasthis LABEL   WORD            ;last history location
        ORG     +2
tmpptr  LABEL   WORD            ;template pointer
        ORG     +2
tmpbeg  LABEL   WORD            ;beginning template offset
        ORG     +2
old1B   LABEL   DWORD           ;original interrupt 1B
        ORG     +4
old21   LABEL   DWORD           ;original interrupt 21
        ORG     +4

;--- display data

page    LABEL   BYTE            ;active page
        ORG     +1
home    LABEL   WORD            ;home location
        ORG     +2
cols    LABEL   BYTE            ;screen columns
        ORG     +1
scroll  LABEL   BYTE            ;horizontal scroll
        ORG     +1
        ORG     +1              ;high byte always zero

;--- input data

inpmax  LABEL   BYTE            ;maximum output bytes
        ORG     +1
inpout  LABEL   DWORD           ;output buffer
        ORG     +4

;--- macro data

tabptr  LABEL   WORD            ;macro table pointer
        ORG     +2
tabbeg  LABEL   WORD            ;beginning of table
        ORG     +2
tabend  LABEL   WORD            ;end of table
        ORG     +2

;--- variable data

varbeg  LABEL   WORD            ;start of variable name buffer
        ORG     +2
varend  LABEL   WORD            ;end of variable name buffer
        ORG     +2
vartab  LABEL   WORD            ;variable nest table
        ORG     +2
varptr  LABEL   WORD            ;variable nest table pointer
        ORG     +2
varsav  LABEL   WORD            ;last variable read
        ORG     +2
varvar  LABEL   WORD            ;last variable pointer
        ORG     +2
varlev  LABEL   BYTE            ;last variable level
        ORG     +1

;--- interpret data

quoflg  LABEL   BYTE            ;quote flag / character
        ORG     +1
bufbeg  LABEL   WORD            ;local input buffer
        ORG     +2
bufend  LABEL   WORD            ;end of data buffer
        ORG     +2
bufptr  LABEL   WORD            ;data pointer
        ORG     +2
parquo  LABEL   BYTE            ;saved quote flag in argument
        ORG     +1
parptr  LABEL   WORD            ;saved pointer in argument
        ORG     +2
partal  LABEL   WORD            ;argument tail
        ORG     +2
lastptr LABEL   WORD            ;last source pointer
        ORG     +2
lastvar LABEL   WORD            ;last variable pointer
        ORG     +2
lastlev LABEL   BYTE            ;last variable level
        ORG     +1
lastquo LABEL   BYTE            ;last quote flag
        ORG     +1

;--- key code data

keybeg  LABEL   WORD            ;keyboard buffer
        ORG     +2
keyend  LABEL   WORD            ;end of keyboard buffer
        ORG     +2
keyptr  LABEL   WORD            ;current pointer
        ORG     +2
keytmp  LABEL   WORD            ;temporary variable
        ORG     +2
keyloo  LABEL   WORD            ;loop count
        ORG     +2

;--- file name completion data

drive   LABEL   BYTE            ;drive letter
        ORG     +1
nambeg  LABEL   WORD            ;start of saved names
        ORG     +2
namend  LABEL   WORD            ;end of name save area
        ORG     +2
namtop  LABEL   WORD            ;end of saved names
        ORG     +2
namptr  LABEL   WORD            ;current name save pointer
        ORG     +2
nampat  LABEL   WORD            ;name pattern buffer address
        ORG     +2
namrec  LABEL   WORD            ;name record address
        ORG     +2
pthend  LABEL   WORD            ;end of path in pattern
        ORG     +2

REL_END

;************************************************************************
; Declared Resident Data

        ORG     FIXORG

;========================================================================
; Windows data structures for private memory blocks. Defined in "PC
; Techniques", FEB/MAR 1991, VOL. 1, NO. 6

;--- block list record

wdata1  LABEL   BYTE
        DB      3, 0                            ;Windows version
        DW      ?, ?                            ;chain to previous block
        DW      0, 0
        DW      0, 0
        DW      OFFSET wdata2, ?                ;pointer to block list (*)

;--- block list

wdata2  LABEL   BYTE
        DW      OFFSET REL_TAR, ?               ;first block address (*)
        DW      OFFSET REL_END - OFFSET REL_TAR ;first block size
        DW      OFFSET RES_END, ?               ;second block address (*)
        DW      ?                               ;second block size (*)
        DW      0, 0                            ;NULL is end of list

;(*) must be initialized before going resident

;========================================================================
; Keystroke assignment data.

;--- keystrokes

_KEY_BKSP       EQU     8
_KEY_TAB        EQU     9
_KEY_ENTER      EQU     13
_KEY_ESC        EQU     27
_KEY_SHF_TAB    EQU     15*256
_KEY_CTL_ENTER  EQU     10
_KEY_CTL_SLASH  EQU     28
_KEY_CTL_MINUS  EQU     31
_KEY_CTL_BKSP   EQU     127
_KEY_ALT_MINUS  EQU     130*256
_KEY_ALT_EQUAL  EQU     131*256
_KEY_CTL_2      EQU     3
_KEY_CTL_6      EQU     30
_KEY_ALT_0      EQU     129*256
_KEY_ALT_1      EQU     120*256
_KEY_ALT_2      EQU     121*256
_KEY_ALT_3      EQU     122*256
_KEY_ALT_4      EQU     123*256
_KEY_ALT_5      EQU     124*256
_KEY_ALT_6      EQU     125*256
_KEY_ALT_7      EQU     126*256
_KEY_ALT_8      EQU     127*256
_KEY_ALT_9      EQU     128*256
_KEY_CTL_A      EQU     1
_KEY_CTL_B      EQU     2
_KEY_CTL_C      EQU     3
_KEY_CTL_D      EQU     4
_KEY_CTL_E      EQU     5
_KEY_CTL_F      EQU     6
_KEY_CTL_G      EQU     7
_KEY_CTL_H      EQU     8
_KEY_CTL_I      EQU     9
_KEY_CTL_J      EQU     10
_KEY_CTL_K      EQU     11
_KEY_CTL_L      EQU     12
_KEY_CTL_M      EQU     13
_KEY_CTL_N      EQU     14
_KEY_CTL_O      EQU     15
_KEY_CTL_P      EQU     16
_KEY_CTL_Q      EQU     17
_KEY_CTL_R      EQU     18
_KEY_CTL_S      EQU     19
_KEY_CTL_T      EQU     20
_KEY_CTL_U      EQU     21
_KEY_CTL_V      EQU     22
_KEY_CTL_W      EQU     23
_KEY_CTL_X      EQU     24
_KEY_CTL_Y      EQU     25
_KEY_CTL_Z      EQU     26
_KEY_ALT_A      EQU     30*256
_KEY_ALT_B      EQU     48*256
_KEY_ALT_C      EQU     46*256
_KEY_ALT_D      EQU     32*256
_KEY_ALT_E      EQU     18*256
_KEY_ALT_F      EQU     33*256
_KEY_ALT_G      EQU     34*256
_KEY_ALT_H      EQU     35*256
_KEY_ALT_I      EQU     23*256
_KEY_ALT_J      EQU     36*256
_KEY_ALT_K      EQU     37*256
_KEY_ALT_L      EQU     38*256
_KEY_ALT_M      EQU     50*256
_KEY_ALT_N      EQU     49*256
_KEY_ALT_O      EQU     24*256
_KEY_ALT_P      EQU     25*256
_KEY_ALT_Q      EQU     16*256
_KEY_ALT_R      EQU     19*256
_KEY_ALT_S      EQU     31*256
_KEY_ALT_T      EQU     20*256
_KEY_ALT_U      EQU     22*256
_KEY_ALT_V      EQU     47*256
_KEY_ALT_W      EQU     17*256
_KEY_ALT_X      EQU     45*256
_KEY_ALT_Y      EQU     21*256
_KEY_ALT_Z      EQU     44*256
_KEY_LEFT       EQU     75*256
_KEY_RIGHT      EQU     77*256
_KEY_UP         EQU     72*256
_KEY_DOWN       EQU     80*256
_KEY_PGUP       EQU     73*256
_KEY_PGDN       EQU     81*256
_KEY_HOME       EQU     71*256
_KEY_END        EQU     79*256
_KEY_CTL_LEFT   EQU     115*256
_KEY_CTL_RIGHT  EQU     116*256
_KEY_CTL_PGUP   EQU     132*256
_KEY_CTL_PGDN   EQU     118*256
_KEY_CTL_HOME   EQU     119*256
_KEY_CTL_END    EQU     117*256
_KEY_INS        EQU     82*256
_KEY_DEL        EQU     83*256
_KEY_F1         EQU     59*256
_KEY_F2         EQU     60*256
_KEY_F3         EQU     61*256
_KEY_F4         EQU     62*256
_KEY_F5         EQU     63*256
_KEY_F6         EQU     64*256
_KEY_F7         EQU     65*256
_KEY_F8         EQU     66*256
_KEY_F9         EQU     67*256
_KEY_F10        EQU     68*256
_KEY_F11        EQU     133*256
_KEY_F12        EQU     134*256
_KEY_SHF_F1     EQU     84*256
_KEY_SHF_F2     EQU     85*256
_KEY_SHF_F3     EQU     86*256
_KEY_SHF_F4     EQU     87*256
_KEY_SHF_F5     EQU     88*256
_KEY_SHF_F6     EQU     89*256
_KEY_SHF_F7     EQU     90*256
_KEY_SHF_F8     EQU     91*256
_KEY_SHF_F9     EQU     92*256
_KEY_SHF_F10    EQU     93*256
_KEY_SHF_F11    EQU     135*256
_KEY_SHF_F12    EQU     136*256
_KEY_CTL_F1     EQU     94*256
_KEY_CTL_F2     EQU     95*256
_KEY_CTL_F3     EQU     96*256
_KEY_CTL_F4     EQU     97*256
_KEY_CTL_F5     EQU     98*256
_KEY_CTL_F6     EQU     99*256
_KEY_CTL_F7     EQU     100*256
_KEY_CTL_F8     EQU     101*256
_KEY_CTL_F9     EQU     102*256
_KEY_CTL_F10    EQU     103*256
_KEY_CTL_F11    EQU     137*256
_KEY_CTL_F12    EQU     138*256
_KEY_ALT_F1     EQU     104*256
_KEY_ALT_F2     EQU     105*256
_KEY_ALT_F3     EQU     106*256
_KEY_ALT_F4     EQU     107*256
_KEY_ALT_F5     EQU     108*256
_KEY_ALT_F6     EQU     109*256
_KEY_ALT_F7     EQU     110*256
_KEY_ALT_F8     EQU     111*256
_KEY_ALT_F9     EQU     112*256
_KEY_ALT_F10    EQU     113*256
_KEY_ALT_F11    EQU     139*256
_KEY_ALT_F12    EQU     140*256

;--- keystroke codes

CMD_BKSP        EQU     1
CMD_TAB         EQU     2
CMD_ENTER       EQU     3
CMD_ESC         EQU     4
CMD_SHF_TAB     EQU     5
CMD_CTL_ENTER   EQU     6
CMD_CTL_SLASH   EQU     7
CMD_CTL_MINUS   EQU     8
CMD_CTL_BKSP    EQU     9
CMD_ALT_MINUS   EQU     10
CMD_ALT_EQUAL   EQU     11
CMD_CTL_2       EQU     12
CMD_CTL_6       EQU     13
CMD_ALT_0       EQU     14
CMD_ALT_1       EQU     15
CMD_ALT_2       EQU     16
CMD_ALT_3       EQU     17
CMD_ALT_4       EQU     18
CMD_ALT_5       EQU     19
CMD_ALT_6       EQU     20
CMD_ALT_7       EQU     21
CMD_ALT_8       EQU     22
CMD_ALT_9       EQU     23
CMD_CTL_A       EQU     24
CMD_CTL_B       EQU     25
CMD_CTL_C       EQU     26
CMD_CTL_D       EQU     27
CMD_CTL_E       EQU     28
CMD_CTL_F       EQU     29
CMD_CTL_G       EQU     30
CMD_CTL_H       EQU     31
CMD_CTL_I       EQU     32
CMD_CTL_J       EQU     33
CMD_CTL_K       EQU     34
CMD_CTL_L       EQU     35
CMD_CTL_M       EQU     36
CMD_CTL_N       EQU     37
CMD_CTL_O       EQU     38
CMD_CTL_P       EQU     39
CMD_CTL_Q       EQU     40
CMD_CTL_R       EQU     41
CMD_CTL_S       EQU     42
CMD_CTL_T       EQU     43
CMD_CTL_U       EQU     44
CMD_CTL_V       EQU     45
CMD_CTL_W       EQU     46
CMD_CTL_X       EQU     47
CMD_CTL_Y       EQU     48
CMD_CTL_Z       EQU     49
CMD_ALT_A       EQU     50
CMD_ALT_B       EQU     51
CMD_ALT_C       EQU     52
CMD_ALT_D       EQU     53
CMD_ALT_E       EQU     54
CMD_ALT_F       EQU     55
CMD_ALT_G       EQU     56
CMD_ALT_H       EQU     57
CMD_ALT_I       EQU     58
CMD_ALT_J       EQU     59
CMD_ALT_K       EQU     60              ;Note: skip uppercase letters
CMD_ALT_L       EQU     123
CMD_ALT_M       EQU     124
CMD_ALT_N       EQU     125
CMD_ALT_O       EQU     126
CMD_ALT_P       EQU     127
CMD_ALT_Q       EQU     128
CMD_ALT_R       EQU     129
CMD_ALT_S       EQU     130
CMD_ALT_T       EQU     131
CMD_ALT_U       EQU     132
CMD_ALT_V       EQU     133
CMD_ALT_W       EQU     134
CMD_ALT_X       EQU     135
CMD_ALT_Y       EQU     136
CMD_ALT_Z       EQU     137
CMD_LEFT        EQU     138
CMD_RIGHT       EQU     139
CMD_UP          EQU     140
CMD_DOWN        EQU     141
CMD_PGUP        EQU     142
CMD_PGDN        EQU     143
CMD_HOME        EQU     144
CMD_END         EQU     145
CMD_CTL_LEFT    EQU     146
CMD_CTL_RIGHT   EQU     147
CMD_CTL_PGUP    EQU     148
CMD_CTL_PGDN    EQU     149
CMD_CTL_HOME    EQU     150
CMD_CTL_END     EQU     151
CMD_INS         EQU     152
CMD_DEL         EQU     153
CMD_F1          EQU     154
CMD_F2          EQU     155
CMD_F3          EQU     156
CMD_F4          EQU     157
CMD_F5          EQU     158
CMD_F6          EQU     159
CMD_F7          EQU     160
CMD_F8          EQU     161
CMD_F9          EQU     162
CMD_F10         EQU     163
CMD_F11         EQU     164
CMD_F12         EQU     165
CMD_SHF_F1      EQU     166
CMD_SHF_F2      EQU     167
CMD_SHF_F3      EQU     168
CMD_SHF_F4      EQU     169
CMD_SHF_F5      EQU     170
CMD_SHF_F6      EQU     171
CMD_SHF_F7      EQU     172
CMD_SHF_F8      EQU     173
CMD_SHF_F9      EQU     174
CMD_SHF_F10     EQU     175
CMD_SHF_F11     EQU     176
CMD_SHF_F12     EQU     177
CMD_CTL_F1      EQU     178
CMD_CTL_F2      EQU     179
CMD_CTL_F3      EQU     180
CMD_CTL_F4      EQU     181
CMD_CTL_F5      EQU     182
CMD_CTL_F6      EQU     183
CMD_CTL_F7      EQU     184
CMD_CTL_F8      EQU     185
CMD_CTL_F9      EQU     186
CMD_CTL_F10     EQU     187
CMD_CTL_F11     EQU     188
CMD_CTL_F12     EQU     189
CMD_ALT_F1      EQU     190
CMD_ALT_F2      EQU     191
CMD_ALT_F3      EQU     192
CMD_ALT_F4      EQU     193
CMD_ALT_F5      EQU     194
CMD_ALT_F6      EQU     195
CMD_ALT_F7      EQU     196
CMD_ALT_F8      EQU     197
CMD_ALT_F9      EQU     198
CMD_ALT_F10     EQU     199
CMD_ALT_F11     EQU     200
CMD_ALT_F12     EQU     201

;--- command codes

CMD_CUR_LEFT    EQU     210
CMD_CUR_RIGHT   EQU     211
CMD_CUR_HOME    EQU     212
CMD_CUR_END     EQU     213
CMD_CUR_PREV    EQU     214
CMD_CUR_NEXT    EQU     215
CMD_DEL_PREV    EQU     216
CMD_DEL_NEXT    EQU     217
CMD_DEL_CHAR    EQU     218
CMD_DEL_FRONT   EQU     219
CMD_DEL_REST    EQU     220
CMD_DEL_LINE    EQU     221
CMD_DEL_BACK    EQU     222
CMD_INS_SET     EQU     223
CMD_INS_CLEAR   EQU     224
CMD_INS_TOGGLE  EQU     225
CMD_HIS_DELETE  EQU     226
CMD_HIS_ADD     EQU     227
CMD_HIS_REVERSE EQU     228
CMD_HIS_FORWARD EQU     229
CMD_HIS_OLDER   EQU     230
CMD_HIS_NEWER   EQU     231
CMD_HIS_TRACE   EQU     232
CMD_HIS_OLDEST  EQU     233
CMD_HIS_NEWEST  EQU     234
CMD_HIS_CLEAR   EQU     235
CMD_HIS_ON      EQU     236
CMD_HIS_OFF     EQU     237
CMD_NAM_MATCH   EQU     238
CMD_NAM_APPEND  EQU     239
CMD_NAM_PREV    EQU     240
CMD_TMP_CHAR    EQU     241
CMD_TMP_INS     EQU     242
CMD_TMP_REM     EQU     243
CMD_TMP_SKIP    EQU     244
CMD_TMP_EOF     EQU     245
CMD_RUN         EQU     246
CMD_RUN_DIRECT  EQU     247
CMD_MASK        EQU     248
CMD_ERROR       EQU     249

;--- key code translation table

keytab  LABEL   WORD
        DW      (OFFSET keytabx - OFFSET keytab) / 3
        DB      CMD_BKSP,       WORD _KEY_BKSP
        DB      CMD_TAB,        WORD _KEY_TAB
        DB      CMD_ENTER,      WORD _KEY_ENTER
        DB      CMD_ESC,        WORD _KEY_ESC
        DB      CMD_SHF_TAB,    WORD _KEY_SHF_TAB
        DB      CMD_CTL_ENTER,  WORD _KEY_CTL_ENTER
        DB      CMD_CTL_SLASH,  WORD _KEY_CTL_SLASH
        DB      CMD_CTL_MINUS,  WORD _KEY_CTL_MINUS
        DB      CMD_CTL_BKSP,   WORD _KEY_CTL_BKSP
        DB      CMD_ALT_MINUS,  WORD _KEY_ALT_MINUS
        DB      CMD_ALT_EQUAL,  WORD _KEY_ALT_EQUAL
        DB      CMD_CTL_2,      WORD _KEY_CTL_2
        DB      CMD_CTL_6,      WORD _KEY_CTL_6
        DB      CMD_ALT_0,      WORD _KEY_ALT_0
        DB      CMD_ALT_1,      WORD _KEY_ALT_1
        DB      CMD_ALT_2,      WORD _KEY_ALT_2
        DB      CMD_ALT_3,      WORD _KEY_ALT_3
        DB      CMD_ALT_4,      WORD _KEY_ALT_4
        DB      CMD_ALT_5,      WORD _KEY_ALT_5
        DB      CMD_ALT_6,      WORD _KEY_ALT_6
        DB      CMD_ALT_7,      WORD _KEY_ALT_7
        DB      CMD_ALT_8,      WORD _KEY_ALT_8
        DB      CMD_ALT_9,      WORD _KEY_ALT_9
        DB      CMD_CTL_A,      WORD _KEY_CTL_A
        DB      CMD_CTL_B,      WORD _KEY_CTL_B
        DB      CMD_CTL_C,      WORD _KEY_CTL_C
        DB      CMD_CTL_D,      WORD _KEY_CTL_D
        DB      CMD_CTL_E,      WORD _KEY_CTL_E
        DB      CMD_CTL_F,      WORD _KEY_CTL_F
        DB      CMD_CTL_G,      WORD _KEY_CTL_G
        DB      CMD_CTL_H,      WORD _KEY_CTL_H
        DB      CMD_CTL_I,      WORD _KEY_CTL_I
        DB      CMD_CTL_J,      WORD _KEY_CTL_J
        DB      CMD_CTL_K,      WORD _KEY_CTL_K
        DB      CMD_CTL_L,      WORD _KEY_CTL_L
        DB      CMD_CTL_M,      WORD _KEY_CTL_M
        DB      CMD_CTL_N,      WORD _KEY_CTL_N
        DB      CMD_CTL_O,      WORD _KEY_CTL_O
        DB      CMD_CTL_P,      WORD _KEY_CTL_P
        DB      CMD_CTL_Q,      WORD _KEY_CTL_Q
        DB      CMD_CTL_R,      WORD _KEY_CTL_R
        DB      CMD_CTL_S,      WORD _KEY_CTL_S
        DB      CMD_CTL_T,      WORD _KEY_CTL_T
        DB      CMD_CTL_U,      WORD _KEY_CTL_U
        DB      CMD_CTL_V,      WORD _KEY_CTL_V
        DB      CMD_CTL_W,      WORD _KEY_CTL_W
        DB      CMD_CTL_X,      WORD _KEY_CTL_X
        DB      CMD_CTL_Y,      WORD _KEY_CTL_Y
        DB      CMD_CTL_Z,      WORD _KEY_CTL_Z
        DB      CMD_ALT_A,      WORD _KEY_ALT_A
        DB      CMD_ALT_B,      WORD _KEY_ALT_B
        DB      CMD_ALT_C,      WORD _KEY_ALT_C
        DB      CMD_ALT_D,      WORD _KEY_ALT_D
        DB      CMD_ALT_E,      WORD _KEY_ALT_E
        DB      CMD_ALT_F,      WORD _KEY_ALT_F
        DB      CMD_ALT_G,      WORD _KEY_ALT_G
        DB      CMD_ALT_H,      WORD _KEY_ALT_H
        DB      CMD_ALT_I,      WORD _KEY_ALT_I
        DB      CMD_ALT_J,      WORD _KEY_ALT_J
        DB      CMD_ALT_K,      WORD _KEY_ALT_K
        DB      CMD_ALT_L,      WORD _KEY_ALT_L
        DB      CMD_ALT_M,      WORD _KEY_ALT_M
        DB      CMD_ALT_N,      WORD _KEY_ALT_N
        DB      CMD_ALT_O,      WORD _KEY_ALT_O
        DB      CMD_ALT_P,      WORD _KEY_ALT_P
        DB      CMD_ALT_Q,      WORD _KEY_ALT_Q
        DB      CMD_ALT_R,      WORD _KEY_ALT_R
        DB      CMD_ALT_S,      WORD _KEY_ALT_S
        DB      CMD_ALT_T,      WORD _KEY_ALT_T
        DB      CMD_ALT_U,      WORD _KEY_ALT_U
        DB      CMD_ALT_V,      WORD _KEY_ALT_V
        DB      CMD_ALT_W,      WORD _KEY_ALT_W
        DB      CMD_ALT_X,      WORD _KEY_ALT_X
        DB      CMD_ALT_Y,      WORD _KEY_ALT_Y
        DB      CMD_ALT_Z,      WORD _KEY_ALT_Z
        DB      CMD_LEFT,       WORD _KEY_LEFT
        DB      CMD_RIGHT,      WORD _KEY_RIGHT
        DB      CMD_UP,         WORD _KEY_UP
        DB      CMD_DOWN,       WORD _KEY_DOWN
        DB      CMD_PGUP,       WORD _KEY_PGUP
        DB      CMD_PGDN,       WORD _KEY_PGDN
        DB      CMD_HOME,       WORD _KEY_HOME
        DB      CMD_END,        WORD _KEY_END
        DB      CMD_CTL_LEFT,   WORD _KEY_CTL_LEFT
        DB      CMD_CTL_RIGHT,  WORD _KEY_CTL_RIGHT
        DB      CMD_CTL_PGUP,   WORD _KEY_CTL_PGUP
        DB      CMD_CTL_PGDN,   WORD _KEY_CTL_PGDN
        DB      CMD_CTL_HOME,   WORD _KEY_CTL_HOME
        DB      CMD_CTL_END,    WORD _KEY_CTL_END
        DB      CMD_INS,        WORD _KEY_INS
        DB      CMD_DEL,        WORD _KEY_DEL
        DB      CMD_F1,         WORD _KEY_F1
        DB      CMD_F2,         WORD _KEY_F2
        DB      CMD_F3,         WORD _KEY_F3
        DB      CMD_F4,         WORD _KEY_F4
        DB      CMD_F5,         WORD _KEY_F5
        DB      CMD_F6,         WORD _KEY_F6
        DB      CMD_F7,         WORD _KEY_F7
        DB      CMD_F8,         WORD _KEY_F8
        DB      CMD_F9,         WORD _KEY_F9
        DB      CMD_F10,        WORD _KEY_F10
        DB      CMD_F11,        WORD _KEY_F11
        DB      CMD_F12,        WORD _KEY_F12
        DB      CMD_SHF_F1,     WORD _KEY_SHF_F1
        DB      CMD_SHF_F2,     WORD _KEY_SHF_F2
        DB      CMD_SHF_F3,     WORD _KEY_SHF_F3
        DB      CMD_SHF_F4,     WORD _KEY_SHF_F4
        DB      CMD_SHF_F5,     WORD _KEY_SHF_F5
        DB      CMD_SHF_F6,     WORD _KEY_SHF_F6
        DB      CMD_SHF_F7,     WORD _KEY_SHF_F7
        DB      CMD_SHF_F8,     WORD _KEY_SHF_F8
        DB      CMD_SHF_F9,     WORD _KEY_SHF_F9
        DB      CMD_SHF_F10,    WORD _KEY_SHF_F10
        DB      CMD_SHF_F11,    WORD _KEY_SHF_F11
        DB      CMD_SHF_F12,    WORD _KEY_SHF_F12
        DB      CMD_CTL_F1,     WORD _KEY_CTL_F1
        DB      CMD_CTL_F2,     WORD _KEY_CTL_F2
        DB      CMD_CTL_F3,     WORD _KEY_CTL_F3
        DB      CMD_CTL_F4,     WORD _KEY_CTL_F4
        DB      CMD_CTL_F5,     WORD _KEY_CTL_F5
        DB      CMD_CTL_F6,     WORD _KEY_CTL_F6
        DB      CMD_CTL_F7,     WORD _KEY_CTL_F7
        DB      CMD_CTL_F8,     WORD _KEY_CTL_F8
        DB      CMD_CTL_F9,     WORD _KEY_CTL_F9
        DB      CMD_CTL_F10,    WORD _KEY_CTL_F10
        DB      CMD_CTL_F11,    WORD _KEY_CTL_F11
        DB      CMD_CTL_F12,    WORD _KEY_CTL_F12
        DB      CMD_ALT_F1,     WORD _KEY_ALT_F1
        DB      CMD_ALT_F2,     WORD _KEY_ALT_F2
        DB      CMD_ALT_F3,     WORD _KEY_ALT_F3
        DB      CMD_ALT_F4,     WORD _KEY_ALT_F4
        DB      CMD_ALT_F5,     WORD _KEY_ALT_F5
        DB      CMD_ALT_F6,     WORD _KEY_ALT_F6
        DB      CMD_ALT_F7,     WORD _KEY_ALT_F7
        DB      CMD_ALT_F8,     WORD _KEY_ALT_F8
        DB      CMD_ALT_F9,     WORD _KEY_ALT_F9
        DB      CMD_ALT_F10,    WORD _KEY_ALT_F10
        DB      CMD_ALT_F11,    WORD _KEY_ALT_F11
        DB      CMD_ALT_F12,    WORD _KEY_ALT_F12
keytabx LABEL   WORD

;--- command code translation table

cmdtab  LABEL   WORD
        DW      (OFFSET cmdtabx - OFFSET cmdtab) / 3
        DB      CMD_CUR_LEFT,   OFFSET Cur_Left
        DB      CMD_CUR_RIGHT,  OFFSET Cur_Right
        DB      CMD_CUR_HOME,   OFFSET Cur_Home
        DB      CMD_CUR_END,    OFFSET Cur_End
        DB      CMD_CUR_PREV,   OFFSET Cur_Prev
        DB      CMD_CUR_NEXT,   OFFSET Cur_Next
        DB      CMD_DEL_PREV,   OFFSET Del_Prev
        DB      CMD_DEL_NEXT,   OFFSET Del_Next
        DB      CMD_DEL_CHAR,   OFFSET Del_Char
        DB      CMD_DEL_FRONT,  OFFSET Clear_Front
        DB      CMD_DEL_REST,   OFFSET Clear_Rest
        DB      CMD_DEL_LINE,   OFFSET Clear_Line
        DB      CMD_DEL_BACK,   OFFSET Backspace
        DB      CMD_INS_SET,    OFFSET Set_Ins
        DB      CMD_INS_CLEAR,  OFFSET Clear_Ins
        DB      CMD_INS_TOGGLE, OFFSET Toggle_Ins
        DB      CMD_HIS_DELETE, OFFSET His_Delete
        DB      CMD_HIS_ADD,    OFFSET His_Add
        DB      CMD_HIS_REVERSE,OFFSET His_Reverse
        DB      CMD_HIS_FORWARD,OFFSET His_Forward
        DB      CMD_HIS_OLDER,  OFFSET His_Older
        DB      CMD_HIS_NEWER,  OFFSET His_Newer
        DB      CMD_HIS_TRACE,  OFFSET His_Trace
        DB      CMD_HIS_OLDEST, OFFSET His_Oldest
        DB      CMD_HIS_NEWEST, OFFSET His_Newest
        DB      CMD_HIS_CLEAR,  OFFSET His_Clear
        DB      CMD_HIS_ON,     OFFSET His_On
        DB      CMD_HIS_OFF,    OFFSET His_Off
        DB      CMD_NAM_MATCH,  OFFSET Name_Match
        DB      CMD_NAM_APPEND, OFFSET Name_Append
        DB      CMD_NAM_PREV,   OFFSET Name_Prev
        DB      CMD_TMP_CHAR,   OFFSET Temp_Char
        DB      CMD_TMP_INS,    OFFSET Temp_Ins
        DB      CMD_TMP_REM,    OFFSET Temp_Rem
        DB      CMD_TMP_SKIP,   OFFSET Temp_Skip
        DB      CMD_TMP_EOF,    OFFSET Temp_EOF
        DB      CMD_RUN,        _CMD_RUN
        DB      CMD_RUN_DIRECT, _CMD_RUN_DIRECT
        DB      CMD_MASK,       _CMD_MASK
cmdtabx LABEL   WORD

;--- special command values used instead of addresses

_CMD_RUN        EQU     0FFFFH
_CMD_RUN_DIRECT EQU     0FFFEH
_CMD_MASK       EQU     0FFFDH

;************************************************************************
; Resident Include Files

        UNUSED-
        INCLUDE 'c:\src\wasm\tsr1'
        INCLUDE 'c:\src\wasm\mulplx1'
        UNUSED+

;************************************************************************
; Interrupt Handlers

;========================================================================
; Break handler.

New1B   PROC    NEAR
        seg     cs
        or      flags, BREAK            ;set break flag
        seg     cs
        jmp     old1B
        ENDP

;========================================================================
; Multiplex interrupt (2F) handler.

New2F   PROC    NEAR

;+++ this input hook may be necessary +++++++++++++++++++++++++++++++++++
;
;;--- input hook
;
;        seg     cs
;        cmp     ax, 4810H               ;DOS input
;        jne     new2Fa
;        seg     cs
;        test    flags, ENABLE           ;check if enabled
;        jz      new2Fa
;        call    Size_Check              ;size check
;        jnz     new2Fa
;
;        pushf
;        push    cs
;        call    Input                   ;input
;        sub     ax, ax                  ;success
;        stc
;        ret
;
;new2Fa

;--- windows entry

        cmp     ax, 1605H               ;window entry message
        jne     new2Fb
        seg     cs
        test    flags4, PRIVATE         ;check if private data
        jz      new2Fb

        seg     cs
        mov     WORD wdata1 + 4, es     ;save chain segment
        seg     cs
        mov     WORD wdata1 + 2, bx     ;save offset
        push    cs
        pop     es                      ;new segment
        mov     bx, OFFSET wdata1       ;new offset

new2Fb  clc
        ret
        ENDP

;========================================================================
; DOS interrupt (21) handler.

New21   seg     cs
        test    flags, ENABLE           ;check if enabled
        jz      new21a
        cmp     ah, 0AH                 ;check if input
        jne     new21a
        call    Size_Check              ;size check
        jz      Input
new21a  seg     cs
        jmp     old21                   ;chain to old interrupt

;========================================================================
; Check if buffer is proper size.
;
; In: Registers setup for input.
;
; Out: ZF= set if size is okay (or /BS not defined).

Size_Check PROC NEAR
        push    ax
        seg     cs
        mov     al, sizchk              ;load size
        or      al, al                  ;check if defined
        jz      sizchk1
        push    bx
        mov     bx, dx
        cmp     al, [bx]                ;check if size matches
        pop     bx
sizchk1 pop     ax
        ret
        ENDP

;************************************************************************
; Line Input

Input   sti                     ;enable interrupts

;========================================================================
; Input line to local buffer.

        push    ax
        push    bx
        push    cx
        push    dx
        push    di
        push    si
        push    bp
        push    ds
        push    es

        mov     bx, dx
        mov     al, [bx]                ;load maximum bytes
        dec     al                      ;don't count CR

        seg     cs
        mov     WORD inpout + 2, ds     ;save segment

        cld
        push    cs
        pop     ds
        push    cs
        pop     es

        inc     dx                      ;skip buffer size
        mov     WORD inpout, dx         ;save offset
        mov     inpmax, al              ;save maximum bytes

;--- set insert

        mov     al, flags               ;load flags
        test    al, IRESET              ;check if reset each entry
        jz      input1
        and     al, NOT INSERT          ;clear flag
        test    al, IVALUE              ;check if set flag
        jz      input1                  ;jump if not
        or      al, INSERT              ;set flag
input1  mov     dl, al
        and     al, NOT BREAK           ;clear break
        mov     flags, al

;--- check if break

        shl     dl                      ;shift break bit into carry
        jnc     input2
        call    Mac_Clear               ;reset all macro data

;--- get video information

input2  mov     ah, 0FH                 ;get video information
        int     10H                     ;execute
        mov     page, bh                ;save page
        push    ax                      ;save columns
        mov     ah, 3                   ;get cursor location
        int     10H                     ;execute
        pop     ax
        mov     home, dx                ;save home position
        sub     ah, dl                  ;calculate available columns
        mov     cols, ah                ;save it

        call    Cur_On                  ;set cursor

;--- prepare for input

        or      flags2, REDISP          ;set flags
        mov     keyloo, MAXLOO          ;reset loop count

        mov     dx, bufbeg              ;start of buffer
        mov     bp, dx                  ;
        mov     di, dx                  ;
        mov     ch, inpmax              ;maximum bytes
        sub     cl, cl                  ;zero current bytes

        mov     ax, loaded              ;preloaded history
        or      ax, ax                  ;check if any
        jz      input3                  ;skip if not

        mov     di, ax                  ;need it in DI
        mov     ax, OFFSET input4       ;return location
        push    ax
        push    ax                      ;dummy value
        sub     ax, ax                  ;clear flags
        jmp     His_Reload              ;load history

input3  xchg    ax, curhis              ;current history entry
        mov     lasthis, ax             ;last history entry
input4  sub     ax, ax                  ;zero AX
        mov     loaded, ax              ;reset preload
        mov     scroll, al              ;zero scroll
        call    Temp_Reset              ;reset template
        mov     al, UPDLIN OR RESET     ;load flags for redraw

;--- check if chain

        mov     bx, flags3              ;load flags
        mov     si, bufptr              ;load pointer
        cmp     si, bufend              ;check if points to end
        je      inputC                  ;jump if not
        and     flags2, NOT REDISP      ;redisplay
        test    bl, RUNCMD              ;check if run command
        jnz     input5                  ;jump if so
        jmp     inter4                  ;branch directly to interpret

input5  mov     bp, bufend              ;write to end of buffer
        and     flags2, NOT REDISP      ;redisplay
        jmp     interP                  ;branch directly to interpret

;=== process input

;--- insert a keystroke, check for ENTER

input6  or      ax, ax                  ;check if null key
        jz      input8                  ;zap input if so
        cmp     ax, _KEY_ENTER          ;check if raw ENTER
        jne     inputA                  ;insert if not
input7  mov     ax, _CMD_RUN            ;substitute run command
        jmps    inputF

;--- null key, abort macros and return empty buffer

input8  call    Mac_Clear               ;cancel macros
        sub     cl, cl                  ;zero input
        jmps    input7

;--- character mask

input9  call    Read_Ctrl               ;read control character
        jc      inputE                  ;skip if command

;--- insert character

inputA  or      ah, ah                  ;check if extended key
        jnz     inputE                  ;skip if so
        and     flags, NOT SECINS       ;clear secondary insert
        mov     ah, al
        cmp     di, dx                  ;check if at end
        jne     inputB
        test    flags, INSERT           ;check if insert
        jnz     inputB
        call    Temp_Next               ;skip to next template character
inputB  sub     al, al                  ;clear flags
        call    Ins_Char                ;insert character

;--- update input

inputC  test    al, RESET               ;check if reset search
        jz      inputD
        mov     search, 0               ;zero search bytes
        mov     rfirst, 0               ;clear first reverse entry found
        mov     ffirst, 0               ;clear first forward entry found

inputD  push    ax
        and     flags2, NOT FBEGUN      ;clear flag
        and     al, HRESET OR FBEGUN    ;mask bits
        or      flags2, al              ;set bits
        pop     ax

        call    Redraw                  ;redraw buffer

;--- get keystroke

inputE  call    Read_Cmd                ;read command
        jnc     input6

inputF  cmp     ax, _CMD_MASK           ;check if mask
        je      input9
        cmp     ax, _CMD_RUN            ;check if run
        je      inputH
        cmp     ax, _CMD_RUN_DIRECT     ;check if run
        je      inputH

;--- process command

        mov     bx, OFFSET inputC       ;load return address
        push    bx                      ;save for return
        push    ax                      ;routine address
        mov     ax, WORD flags2 - 1     ;load flags and clear AL (NOTE_A)
        mov     bx, hisbeg              ;first string
        push    di
        mov     di, hisend              ;end address
        cmp     di, bx                  ;check if no history
        je      inputG
        call    Str_Prev                ;get last string
inputG  mov     si, di                  ;last string in SI
        pop     di
        retn

;--- finish with input

inputH  push    ax
        mov     bx, dx
        mov     BYTE [bx], 0            ;store NUL (for copying to history)

        mov     al, CLRCUR OR UPDCUR
        call    Redraw                  ;clear cursor block
        mov     ax, cexit               ;exit pattern
        call    Cur_Set                 ;set cursor

        mov     si, bp                  ;start of buffer
        sub     ch, ch                  ;zero high byte of length

;========================================================================
; Copy input to history.

        jcxz    inter1                  ;skip copy if length 0
        test    flags4, HLOCK OR HSKIP  ;check if locked
        jnz     inter1                  ;skip copy if so

        cmp     cl, minlen              ;check if below minimum length
        jb      inter1
        mov     ah, flags2              ;load flags
        
;--- check if delete all identical entries

        test    ah, PURHIS              ;check if pure history
        jz      copy3
copy1   call    His_Find                ;search for match
        jnc     copy5
        mov     bx, curhis              ;load current history
        or      bx, bx                  ;check if none
        jz      copy2
        cmp     di, bx                  ;check if deleteing below
        jae     copy2
        inc     cx                      ;include NUL
        sub     bx, cx                  ;adjust current
        dec     cx                      ;restore length
copy2   mov     curhis, di              ;set for delete
        push    ax
        and     ah, NOT HRESET          ;always delete
        call    His_Remove              ;delete entry
        pop     ax
        mov     curhis, bx              ;save current
        jmps    copy1

;--- check if delete from stack

copy3   test    ah, DELHIS              ;check if remove
        jz      copy4
        call    His_Remove              ;remove it

;--- check if unique input only

copy4   test    ah, UNIQUE              ;check if unique input only
        jz      copy5
        call    His_Find                ;search history
        jc      inter1

;--- add to table

copy5   mov     di, bp
        call    Heap_Add                ;add to history

;========================================================================
; Interpret input line and copy to buffer.

;=== check if run direct

inter1  pop     ax
        call    His_On                  ;reset skip
        cmp     ax, _CMD_RUN_DIRECT     ;check if skip save
        jne     inter2
        mov     curhis, 0               ;zero current
        jmp     interK                  ;go copy to buffer

inter2  mov     bx, flags3              ;load flags

;--- copy to head of stream

inter3  add     si, cx                  ;point to end
        dec     si                      ;last byte
        mov     di, bufptr              ;start of stream
        dec     di                      ;previous byte
        std
        rep
        movsb                           ;pre-pend to stream
        cld
        inc     di                      ;point to first byte
        mov     si, di

inter4  mov     bp, si                  ;stream head pointer
        mov     di, bufbeg              ;start of buffer
        sub     dx, dx                  ;no arguments (not expanding macro)
        or      bl, QUOTE               ;absorb quotes
        push    di

;--- skip initial delimiters

inter5  call    Src_Read                ;read character
        jz      inter9
        cmp     al, DELIM               ;check if delimiter
        je      inter5
        cmp     al, liter               ;check if literal
        jne     inter8
        mov     bp, si                  ;set stream pointer
        jmps    interE

;--- error

inter6  pop     ax                      ;error (from below), fix stack
inter7  call    Mac_Clear               ;reset data
        call    Beep                    ;beep
        sub     cl, cl                  ;zero length
        jmp     interL

;--- copy first field

inter8  cmp     di, bp                  ;check if out of space
        je      inter6
        stosb                           ;store byte
        call    Src_Read                ;read next byte
        jz      inter9
        cmp     al, DELIM               ;check if end of field
        jne     inter8

;--- check if macro

inter9  call    Src_Back                ;reread last character
        mov     quoflg, 0               ;reset quote flag
        mov     bp, si                  ;new stream top

        pop     si                      ;start of field
        mov     cx, di                  ;end of field
        sub     cx, si                  ;calculate length
        push    di
        jz      interE                  ;skip if none

        mov     di, tabptr              ;macro pointer
        mov     di, [di+2]              ;load starting search address
        or      di, di                  ;check if any
        jz      interA
        call    Str_Comp                ;check if in same macro
        je      interC
interA  mov     di, macbeg              ;start of macros
        jmps    interD
interB  call    Str_Comp                ;compare macros
        je      interO
interC  call    Str_Next                ;skip macro name
        call    Str_Next                ;skip replacement text
interD  cmp     di, macend              ;check if at end
        jne     interB

;--- copy remaining line

interE  pop     di
        mov     si, bp                  ;remainder of fields
        jmps    interG
interF  stosb                           ;store byte
interG  call    Src_Read                ;read a character
        jnz     interF
        or      al, al                  ;check if chain
        jnz     interH
        call    Src_Back                ;don't skip over NUL

interH  mov     bufptr, si              ;save top of stream
        mov     flags3, bx              ;save flags

;--- check if pop any macro levels

        mov     bx, tabptr              ;active macro
        jmps    interJ
interI  sub     bx, 4                   ;previous entry
interJ  cmp     si, [bx]                ;check if should pop macro
        jae     interI
        mov     tabptr, bx              ;new macro pointer

;--- copy to input buffer

        mov     si, bufbeg              ;start of buffer
        mov     cx, di                  ;end of output
        sub     cx, si                  ;bytes to copy

interK  cmp     cl, [inpmax]            ;check if too large
        ja      interN

interL  les     di, inpout              ;output buffer
        mov     al, cl
        stosb                           ;store length
        push    di
        rep
        movsb                           ;copy data
        mov     al, 13
        stosb                           ;store carriage return
        pop     di

;--- display if chained command

        test    flags2, REDISP          ;check if redisplay
        jnz     interM
        call    Display                 ;display line

;--- finished

interM  mov     ah, 2                   ;output function
        mov     dl, 13                  ;carriage return
        int     21H                     ;display

        pop     es
        pop     ds
        pop     bp
        pop     si
        pop     di
        pop     dx
        pop     cx
        pop     bx
        pop     ax
        iret

interN  jmp     inter7                  ;error relay point

;--- macro, copy to workspace while expanding

interO  pop     ax                      ;fix stack
        mov     si, tabptr              ;load macro table pointer
        cmp     si, tabend              ;check if full
        je      interN
        add     si, 4                   ;next entry
        mov     [si+2], di              ;save macro name
        mov     tabptr, si              ;save pointer

        call    Str_Next                ;skip to data
        mov     si, di

interP  mov     dx, bp                  ;data for parameters
        mov     di, bufbeg              ;start of buffer
        and     bl, NOT QUOTE           ;copy quotes
        jmps    interR

interQ  cmp     di, bp                  ;check if buffer full
        je      interN
        stosb                           ;store byte
        cmp     al, chain               ;check if chain
        jne     interS
interR  mov     partal, bp              ;reset tail
        and     bl, NOT SRCEOL          ;clear end of line flag
interS  call    Src_Read                ;read next byte
        or      al, al                  ;check if NUL
        jnz     interQ                  ;loop back if not

        mov     ax, bp                  ;end of buffer if run command
        test    bl, RUNCMD              ;check if run command
        jnz     interT

        mov     ax, partal
        mov     si, tabptr              ;load macro table pointer
        mov     [si], ax                ;save macro scope

interT  mov     bufptr, ax              ;stream head
        mov     si, bufbeg
        mov     cx, di                  ;end of data
        sub     cx, si                  ;calculate length
        and     bl, NOT RUNCMD          ;clear run command
        jmp     inter3

;************************************************************************
; Standard Routines

;========================================================================
; Convert a letter to lower case.
;
; In: AL= letter to convert.
;
; Out: AL= converted.

To_Lower PROC   NEAR
        cmp     al, 'A'                 ;check if too low
        jb      tolow1
        cmp     al, 'Z'                 ;check if too high
        ja      tolow1
        add     al, 20H                 ;convert to lower case
tolow1  ret
        ENDP

;========================================================================
; Convert a letter to upper case.
;
; In: AL= letter to convert.
;
; Out: AL= converted.

To_Upper PROC   NEAR
        cmp     al, 'a'                 ;check if too low
        jb      toupp1
        cmp     al, 'z'                 ;check if too high
        ja      toupp1
        sub     al, 20H                 ;convert to upper case
toupp1  ret
        ENDP

;========================================================================
; Check if character is a delimiter.
;
; In: DI= location of character.
;
; Out: CY= set if not a delimiter.

Is_Delim PROC   NEAR
        push    ax
        push    cx
        push    di
        mov     al, [di]                ;load byte
        mov     cx, dlmlen              ;number of delimiters
        mov     di, OFFSET dlmchr       ;delimiter list
        repne
        scasb                           ;scan for delimiter
        je      isdel1
        stc                             ;not a delimiter
isdel1  pop     di
        pop     cx
        pop     ax
        ret

;--- delimiter list, all characters that can't be in a filspec

dlmchr  DB      ' "+/;<>=+[]|'
dlmlen  EQU     $ - OFFSET dlmchr
        ENDP

;========================================================================
; Get previous string.  There must be a NUL in front of the previous
; string.
;
; In: DI= starting location.

Str_Prev PROC   NEAR
        dec     di                      ;end of previous string
strpre1 dec     di                      ;decrement pointer
        cmp     BYTE [di], 0            ;check if end of string before prev
        jne     strpre1
        inc     di                      ;start of string
        ret
        ENDP

;========================================================================
; Skip a string.
;
; In: DI= string.
;
; Out: DI= next string.

        PROC    NEAR
strnex1 inc     di                      ;increment pointer
Str_Next LABEL NEAR                     ;entry point
        cmp     BYTE [di], 0            ;check if end of string
        jne     strnex1
        inc     di                      ;start of string
        ret
        ENDP

;========================================================================
; Length of string.
;
; In: DI= string.
;
; Out: CX= length (not including NUL); ZF= set if zero length string.

Str_Length PROC NEAR
        push    di
        mov     cx, di                  ;save location
        call    Str_Next                ;end of string
        xchg    cx, di
        sub     cx, di                  ;calculate bytes
        dec     cx                      ;adjust
        pop     di
        ret
        ENDP

;========================================================================
; Compare characters.
;
; In: SI= source; DI= destination; CX= bytes to compare.
;
; Out: ZF= set if same.
;
; Note: if characters match (ZF=1), the carry flag will always be
; cleared, BUT the carry flag cannot be used to determine a match (i.e.
; the state of the carry flag is indeterminate if if the strings don't
; match).

Chr_Comp PROC   NEAR
        push    ax
        push    cx
        push    di
        push    si
chrcmp1 lodsb                           ;load source byte
        test    flags, CASE
        jnz     chrcmp2
        call    To_Upper                ;convert to upper case
chrcmp2 mov     ah, al
        mov     al, [di]                ;load destination byte
        inc     di                      ;increment pointer
        test    flags, CASE
        jnz     chrcmp3
        call    To_Upper                ;convert to upper case
chrcmp3 cmp     al, ah                  ;check if bytes the same
        jne     chrcmp4
        loop    chrcmp1                 ;loop for all bytes
chrcmp4 pop     si
        pop     di
        pop     cx
        pop     ax
        ret
        ENDP

;========================================================================
; Compare two strings.  The destination string must end with a NUL (the
; source length is passed in CX and the source string may or may not end
; in a NUL).
;
; In: SI= source; DI= destination; CX= length of source string.
;
; Out: ZF= set if match.

Str_Comp PROC   NEAR
        call    Chr_Comp                ;compare characters
        jne     strcmp1                 ;exit if no match
        push    di
        add     di, cx                  ;point to end of characters
        cmp     BYTE [di], 0            ;make sure destination ends
        pop     di
strcmp1 ret
        ENDP

;========================================================================
; Perform a search.  The number of bytes in 'search' are compared.
;
; In: SI= target string; DI= starting string; AX= switch routine; BX=
;     ending string.
;
; Out: CY= set if not found; DI= start of string if found.

        PROC    NEAR
strsea1 call    ax                      ;next string
Str_Search LABEL NEAR
        push    cx
        mov     cx, WORD search         ;bytes to compare
        cmp     cx, 0FFH                ;check if 0 byte search
        je      strsea2
        call    Chr_Comp                ;compare strings
strsea2 pop     cx
        je      strsea3                 ;exit if match
        cmp     di, bx                  ;check if more strings
        jne     strsea1                 ;loop back if so
        stc
strsea3 ret
        ENDP

;========================================================================
; Release heap space.
;
; In: CX= bytes needed.
;
; Out: CY= set if unavailable.

Heap_Rel PROC   NEAR
        push    ax
        push    di
        push    si
        mov     ax, macbeg              ;start of macros
        sub     ax, hisbeg              ;get maximum free bytes
        cmp     ax, cx                  ;check if enough
        jb      hearel5
        jmps    hearel4

hearel1 push    cx
        mov     di, hisbeg              ;start of history (start of heap)
        mov     si, di
        call    Str_Next                ;next string
        xchg    si, di                  ;swap, copy over first string

        mov     ax, lasthis             ;load last entry
        sub     ax, si                  ;get offset from start
        jb      hearel2
        add     ax, di                  ;new address
        jmps    hearel3
hearel2 sub     ax, ax                  ;zero last entry
hearel3 mov     lasthis, ax             ;save new address

        mov     cx, hisend              ;end of history
        sub     cx, si                  ;get bytes to shift
        rep
        movsb                           ;copy bytes
        mov     hisend, di              ;save new end
        pop     cx

hearel4 mov     ax, macbeg              ;start of macros
        sub     ax, hisend              ;get free bytes
        cmp     ax, cx                  ;check if enough
        jb      hearel1

hearel5 pop     si
        pop     di
        pop     ax
        ret
        ENDP

;========================================================================
; Add string to the history.  String isn't saved if identical to last
; string.
;
; In: DI= string.
;
; Out: CY= set if insufficient heap.

Heap_Add PROC   NEAR
        push    cx
        push    si
        call    Str_Length              ;get string length
        inc     cx                      ;include NUL
        call    Heap_Rel                ;release heap space
        jc      heaadd2                 ;exit if space unavailable

        mov     si, di
        mov     di, hisend              ;history end
        cmp     di, hisbeg              ;check if any strings defined
        je      heaadd1
        dec     cx                      ;don't count NUL in compare
        push    di
        call    Str_Prev                ;get previous entry
        call    Str_Comp                ;compare strings
        pop     di
        je      heaadd2
        inc     cx
heaadd1 rep
        movsb                           ;copy to history
        mov     hisend, di              ;new end
heaadd2 pop     si
        pop     cx
        ret
        ENDP

;========================================================================
; Search for history.
;
; In: SI= line; CX= length.
;
; Out: DI= start of macro; CY= set if found.

His_Find PROC   NEAR
        mov     di, hisbeg              ;start of history
        jmps    hisfin2
hisfin1 call    Str_Comp                ;check if match
        stc
        je      hisfin3
        call    Str_Next                ;next entry
hisfin2 cmp     di, hisend              ;check if end
        jne     hisfin1
hisfin3 ret
        ENDP

;========================================================================
; Search for macro
;
; In: SI= macro; CX= length.
;
; Out: DI= start of macro; CY= set if found.

Mac_Find PROC   NEAR
        mov     di, macbeg              ;start of macros
        jmps    macfin2
macfin1 call    Str_Comp                ;compare macros
        je      macfin3
        call    Str_Next                ;skip macro name
        call    Str_Next                ;skip replacement text
macfin2 cmp     di, macend              ;check if at end
        jne     macfin1
        clc
        ret
macfin3 stc
        ret
        ENDP

;************************************************************************
; Input Routines

;========================================================================
; Read a keystroke.
;
; Out: AX= keystroke.

Read_Key PROC   NEAR
        mov     ah, keyfun              ;read function
        int     21H                     ;execute
        sub     ah, ah                  ;zero high byte
        or      al, al                  ;check if extended character
        jnz     reakey1                 ;exit if so
        mov     ah, keyfun              ;read function
        int     21H                     ;execute
        mov     ah, al                  ;put in high byte
        sub     al, al                  ;zero low byte
reakey1 ret
        ENDP

;========================================================================
; Read a control key.
;
; Out: AL= control code; CY= set if invalid key.

Read_Ctrl PROC  NEAR
        call    Read_Cmd                ;get another keystroke
        jc      reacnt1                 ;ignore if command
        and     al, NOT 20H             ;convert to uppercase
        sub     al, 'A' - 1             ;convert to control character
        jbe     reacnt1                 ;ignore if too small
        cmp     al, 31
        ja      reacnt1                 ;ignore if too big
        clc
        ret
reacnt1 stc
        ret
        ENDP

;================================================================================
; Search key table.
;
; In: AX= keystroke;
;
; Out: AL= code; CY= set if found.

Search_Key PROC NEAR
        mov     bx, OFFSET keytab       ;start of table
        mov     cx, [bx]
        inc     bx
        inc     bx

seakey1 cmp     ax, [bx + 1]            ;check if match
        je      seakey2
        add     bx, 3                   ;next entry
        loop    seakey1                 ;loop while entries
        clc
        ret

seakey2 mov     al, [bx]                ;return code
        stc
        ret
        ENDP

;================================================================================
; Search a code table.
;
; In: AL= code; BX= start of table.
;
; Out: AX= translation; CY= set if found.

Search_Cmd PROC NEAR
        mov     cx, [bx]                ;load table entries
        inc     bx
        inc     bx

seacmd1 cmp     al, [bx]                ;check if match
        je      seacmd2
        add     bx, 3                   ;next entry
        loop    seacmd1                 ;loop while entries

        clc
        ret

seacmd2 inc     bx                      ;advance to code
        mov     ax, [bx]                ;return code
        stc
        ret
        ENDP

;================================================================================
; Input a keycode.
;
; Out: AX= key code; CY= set if code, clear if ASCII.

Input_Cmd PROC  NEAR
        call    Read_Key                ;get a real key
        call    Search_Key              ;search code table
        ret
        ENDP

;================================================================================
; Read a command.
;
; Out: AX= ASCII code or command; CY= set if AX is command.

Read_Cmd PROC   NEAR
        push    cx
        push    di

reacmd1 test    flags, BREAK            ;check if break
        jnz     reacmd8
        dec     keyloo                  ;check if too many iterations
        jz      reacmd7
        mov     si, keyptr              ;load pointer
        cmp     si, keybeg              ;check if buffer empty
        je      reacmd3

;--- load key code from buffer

        dec     si
        dec     keyptr
        mov     al, [si]                ;load byte
        sub     ah, ah                  ;zero high byte
        cmp     al, COMMAND             ;check if command
        jne     reacmd4                 ;jump if not
        cmp     si, keybeg              ;check if buffer empty
        je      reacmd1
        dec     si
        dec     keyptr
        mov     al, [si]                ;load command code

;--- key code, check if macro assigned

reacmd2 mov     si, OFFSET keytmp       ;temporary variable
        mov     ah, COMMAND             ;command delimiter
        mov     [si], ax                ;save string
        mov     cx, 2                   ;length
        call    Mac_Find                ;search for macro
        jc      reacmd6

;--- not a macro, check if command

        mov     bx, OFFSET cmdtab       ;command table
        call    Search_Cmd              ;search
        jc      reacmd5

;--- not a command, convert to keystroke

        mov     bx, OFFSET keytab       ;command table
        call    Search_Cmd              ;search
        jnc     reacmd1                 ;unassociated command, skip it
        jmps    reacmd4

;--- input a key code

reacmd3 call    Input_Cmd               ;input a command
        jc      reacmd2                 ;jump if code

reacmd4 clc
reacmd5 pop     di
        pop     cx
        ret

;--- found assigned macro, copy to buffer

reacmd6 call    Str_Next                ;skip macro name
        call    Str_Length              ;get length
        mov     si, di
        mov     di, keyptr              ;current location
        mov     ax, keyend              ;end of buffer
        sub     ax, di                  ;calculate available space
        cmp     cx, ax                  ;check if too big
        ja      reacmd7                 ;jump if too big
        rep
        movsb                           ;copy to buffer
        mov     keyptr, di              ;save new pointer
        jmps    reacmd1

;--- error

reacmd7 call    Beep                    ;beep
reacmd8 call    Mac_Clear               ;stop everything
        and     flags, NOT BREAK        ;clear break
        jmp     reacmd1
        ENDP

;************************************************************************
; Display Routines

;========================================================================
; Execute an interrupt 10H.  Save the registers DI, SI, and BP.

Int10   PROC    NEAR
        push    di
        push    si
        push    bp
        int     10H
        pop     bp
        pop     si
        pop     di
        ret
        ENDP

;========================================================================
; Beep speaker.

Beep    PROC    NEAR
        mov     ax, 0E07H               ;TTY beep
        call    int10                   ;execute
        ret
        ENDP

;========================================================================
; Position the cursor.
;
; In: DL= relative column number.
;
; Notes: preserves AX and DX.

Move_Cur PROC   NEAR
        push    ax
        push    dx
        mov     ah, 2                   ;set cursor function
        sub     dh, dh                  ;clear row offset
        add     dx, home                ;calulate location
        mov     bh, page                ;video page
        call    Int10                   ;execute
        pop     dx
        pop     ax
        ret
        ENDP

;========================================================================
; Set cursor type.
;
; In: AX= cursor scan pattern.

Cur_Set PROC    NEAR
        test    flags, CURSOR           ;check if setting cursor
        jz      curset2
        push    cx
        cmp     ax, DCURSOR             ;check if use default
        jne     curset1
        mov     ax, cdef                ;load default
curset1 mov     cx, ax
        mov     ah, 1                   ;set cursor function
        call    Int10                   ;execute
        pop     cx
curset2 ret
        ENDP

;========================================================================
; Turn the cursor on.

Cur_On  PROC    NEAR
        mov     ax, cins                ;insert scan pattern
        test    flags, INSERT           ;check if insert mode
        jnz     curon1
        mov     ax, cover               ;overwrite scan pattern
curon1  call    Cur_Set                 ;set cursor
        ret
        ENDP

;========================================================================
; Display a character.
;
; In: AL= character; BL= attribute; DL= relative column number.

Draw_Char PROC  NEAR
        push    cx
        push    dx
        call    Move_Cur                ;set position
        mov     ah, 9                   ;function 9
        mov     bh, page                ;video page
        mov     cx, 1                   ;one character
        call    Int10                   ;execute
        pop     dx
        pop     cx
        ret
        ENDP

;========================================================================
; Clear some number of columns using the end attribute.
;
; In: DL= relative column number; CX= columns.

Draw_Clear PROC NEAR
        call    Move_Cur                ;set position
        mov     ax, 0920H               ;display blank
        mov     bl, aend                ;end attribute
        mov     bh, page                ;video page
        call    Int10                   ;execute
        ret
        ENDP

;========================================================================
; Redraw the buffer.
;
; In: AL= flags.

Redraw  PROC    NEAR
        push    cx
        push    bp
        mov     si, bp                  ;start of buffer
        mov     bp, dx                  ;save end of data in BP

        cmp     di, oldloc              ;check if new location
        je      redraw1                 ;skip if not
        or      al, UPDCUR OR UPDOLD    ;update this location

redraw1 mov     cl, cols                ;load column count
        sub     ch, ch
        sub     dl, dl                  ;display column number
        mov     dh, al                  ;save flags in DH

;--- reset scroll

        mov     ax, di
        sub     ax, si                  ;offset from start
        cmp     al, scroll              ;check if left off edge
        jae     redraw2
        mov     scroll, al              ;reset scroll
        or      dh, UPDLIN              ;update whole line

redraw2 sub     al, scroll              ;columns past left edge
        cmp     al, cl                  ;check if past edge
        jb      redraw3
        sub     al, cl                  ;columns to shift
        inc     al
        add     scroll, al              ;adjust scroll
        or      dh, UPDLIN              ;update whole line

;=== loop for each column

redraw3 add     si, WORD scroll         ;skip scroll
        push    si                      ;save for after loop

;--- check if past end

redraw4 cmp     si, bp                  ;check if past end
        ja      redrawD

;--- load attribute

        test    dh, CLRCUR              ;check if clear cursor
        jnz     redraw5
        mov     bl, acurs               ;cursor attribute
        cmp     si, di
        je      redraw6
redraw5 mov     bl, atext               ;text attribute
        cmp     si, bp
        jb      redraw6
        mov     bl, aend                ;end attribute

;--- load character

redraw6 mov     al, ' '                 ;use space if at end
        cmp     si, bp                  ;check if past end
        je      redraw7
        mov     al, [si]                ;load byte

;--- determine if this character should be displayed

redraw7 test    dh, UPDLIN              ;check if redraw whole line
        jnz     redrawA

        test    dh, UPDRIG              ;check if update to right
        jz      redraw8
        cmp     si, di                  ;check if to right
        jae     redrawA

redraw8 test    dh, UPDCUR              ;check if cursor location
        jz      redraw9
        cmp     si, di                  ;check if over cursor
        je      redrawA

redraw9 test    dh, UPDOLD              ;check if cursor location
        jz      redrawB
        cmp     si, oldloc              ;check if over old location
        jne     redrawB

;--- display character

redrawA call    Draw_Char               ;display character
redrawB inc     dl                      ;increment column
        inc     si                      ;increment pointer
        loop    redraw4                 ;loop for each column

;=== finished

redrawC pop     si

        mov     dx, di
        sub     dx, si
        call    Move_Cur                ;position physical cursor

        mov     oldloc, di              ;save current location
        mov     dx, bp
        pop     bp
        pop     cx
        ret

;--- end spaces

redrawD test    dh, UPDLIN OR UPDRIG    ;check if update
        jz      redrawC
        call    Draw_Clear              ;clear line
        jmps    redrawC
        ENDP

;========================================================================
; Display an input buffer (for chained commands). The input line should
; end with a CR.
;
; In: ES:DI= start of input buffer data.

Display PROC    NEAR
        mov     cl, cols                ;load column count
        sub     ch, ch
        sub     dl, dl                  ;display column number

displ1  seg     es
        mov     al, [di]                ;load byte
        cmp     al, 13                  ;check if end of line
        je      displ2
        mov     bl, atext               ;load attribute
        call    Draw_Char               ;display character
        inc     dl                      ;increment column
        inc     di                      ;increment pointer
        loop    displ1                  ;loop for each column
        ret

;--- end spaces

displ2  call    Draw_Clear              ;clear rest of line
        ret
        ENDP

;************************************************************************
; Editing Routines
;
; Register Usage:
;
;   AL  flags
;   BP  start of buffer
;   DX  end of buffer
;   DI  cursor location in buffer
;   CH  maximum bytes in buffer
;   CL  current bytes in buffer

;========================================================================
; Move the cursor left.

Cur_Left PROC   NEAR
        cmp     di, bp                  ;check if start of buffer
        je      curlef1
        dec     di                      ;decrement column
curlef1 ret
        ENDP

;========================================================================
; Move the cursor right.

Cur_Right PROC  NEAR
        cmp     di, dx                  ;check if end of buffer
        je      currig1
        inc     di                      ;increment column
currig1 ret
        ENDP

;========================================================================
; Move the cursor to the beginning of the line.

Cur_Home PROC   NEAR
        mov     di, bp                  ;goto beginning of buffer
        ret
        ENDP

;========================================================================
; Move the cursor to the end of the line.

Cur_End PROC    NEAR
        mov     di, dx                  ;goto end of buffer
        ret
        ENDP

;========================================================================
; Move the cursor to the left word.

Cur_Prev PROC   NEAR                    ;move cursor

;--- find non-delimiter

curpre1 cmp     di, bp                  ;check if at start
        je      curpre3
        dec     di                      ;move left
        call    Is_Delim                ;check if delimiter
        jnc     curpre1

;--- find delimiter

curpre2 cmp     di, bp                  ;check if at start
        je      curpre3
        dec     di                      ;move left
        call    Is_Delim                ;check if delimiter
        jc      curpre2
        inc     di

curpre3 ret
        ENDP

;========================================================================
; Move the cursor to the right word.

        PROC   NEAR                     ;move cursor

;--- find delimiter

curnex1 inc     di                      ;move right
Cur_Next LABEL  NEAR
        cmp     di, dx                  ;check if at end
        je      curnex3
        call    Is_Delim                ;check if delimiter
        jc      curnex1

;--- find non-delimiter

curnex2 inc     di                      ;move right
        cmp     di, dx                  ;check if at end
        je      curnex3
        call    Is_Delim                ;check if delimiter
        jnc     curnex2

curnex3 ret
        ENDP

;========================================================================
; Move the cursor left and delete.

Backspace PROC  NEAR
        cmp     di, bp                  ;check if start of buffer
        je      bacspc2
        cmp     di, dx                  ;check if end of buffer
        jb      bacspc1
        mov     bx, tmpptr              ;load template pointer
        or      bx, bx                  ;check if defined
        jz      bacspc1
        cmp     bx, tmpbeg              ;check if at start of template
        je      bacspc1
        dec     tmpptr                  ;decrement pointer
bacspc1 dec     di                      ;move left
        call    Del_Char                ;delete character
bacspc2 ret
        ENDP

;========================================================================
; Clear all characters to left of cursor.

        PROC NEAR
clrfro1 dec     di                      ;move left
        call    Del_Char                ;delete character
Clear_Front LABEL NEAR                  ;entry point
        cmp     di, bp                  ;check if at edge
        jne     clrfro1                 ;loop back if not
        ret
        ENDP

;========================================================================
; Clear the entire line.

Clear_Line PROC NEAR
        mov     curhis, 0               ;zero current
        call    Cur_Home                ;move cursor home
        call    Clear_Rest              ;clear rest of line
        call    Temp_Reset              ;reset template
        ret
        ENDP

;========================================================================
; Clear from the cursor.

Clear_Rest PROC NEAR
        mov     dx, di                  ;set end to cursor
        mov     bx, di                  ;cursor location
        sub     bx, bp                  ;get bytes in buffer
        mov     cl, bl                  ;save it
        or      al, UPDRIG OR RESET OR HRESET ;flags
        ret
        ENDP

;========================================================================
; Delete the word to the left.

Del_Prev PROC   NEAR
        mov     si, di                  ;save current location
        call    Cur_Prev                ;goto previous word
        sub     si, di                  ;get bytes to delete
        jz      delpre2
delpre1 push    si
        call    Del_Char                ;delete character
        pop     si
        dec     si                      ;decrement count
        jnz     delpre1
delpre2 ret
        ENDP

;========================================================================
; Delete the word to the right.

Del_Next PROC   NEAR
        mov     si, di                  ;save current location
        call    Cur_Next                ;goto next word
        sub     si, di                  ;get bytes to delete
        jz      delnex2
        neg     si                      ;adjust sign
delnex1 push    si
        call    BackSpace               ;delete previous character
        pop     si
        dec     si                      ;decrement count
        jnz     delnex1
delnex2 ret
        ENDP

;========================================================================
; Reset template pointer.

Temp_Reset PROC NEAR
        push    di
        sub     bx, bx
        mov     di, curhis              ;load history pointer
        or      di, di                  ;check if scrolled
        jnz     temres1
        mov     di, hisend              ;load end
temres1 cmp     di, hisbeg              ;check if at start
        je      temres2
        call    Str_Prev                ;goto previous entry
        mov     bx, di
temres2 mov     tmpbeg, bx              ;save template beginning
        mov     tmpptr, bx              ;save template pointer
        pop     di
        ret
        ENDP

;========================================================================
; Insert EOF character.

Temp_EOF PROC   NEAR
        mov     ah, 26
        call    Ins_Char
        ret
        ENDP

;========================================================================
; Move the cursor right.

Temp_Char PROC  NEAR
        cmp     di, dx                  ;check if end of buffer
        je      tmpchr1
        inc     di                      ;increment column
        ret

tmpchr1 mov     bx, tmpptr              ;load template pointer
        or      bx, bx                  ;check if any
        jz      tmpchr2
        mov     ah, [bx]                ;load byte from template
        or      ah, ah                  ;ignore if end of line
        jz      tmpchr2
        call    Ins_Char                ;insert character
        inc     tmpptr                  ;increment pointer
tmpchr2 ret
        ENDP

;========================================================================
; Skip to next template character.

Temp_Next PROC  NEAR
        mov     bx, tmpptr              ;load template pointer
        or      bx, bx                  ;check if defined
        jz      temdel1
        cmp     BYTE [bx], 0            ;check if end of template
        je      temdel1
        inc     tmpptr                  ;increment template pointer
temdel1 ret
        ENDP

;========================================================================
; Copy remaining template.

Temp_Rem PROC   NEAR                    ;entry point
        mov     bx, tmpptr              ;load template pointer
        or      bx, bx                  ;check if defined
        jnz     tmprem2
        ret
tmprem1 inc     bx                      ;increment pointer
        call    Ins_Char                ;insert character
tmprem2 mov     ah, [bx]                ;load next character
        or      ah, ah                  ;check if end of line
        jnz     tmprem1
        mov     tmpptr, bx              ;save pointer
        or      al, UPDLIN              ;update whole line
        ret
        ENDP

;========================================================================
; Input a key and scan the template.
;
; Out: BX= template location; CY= invalid key, no template defined, or
;      character not found.

Temp_Scan PROC  NEAR
        push    ax
        call    Read_Cmd                ;read a keystroke
        jnc     tmpsca1
        cmp     ax, _CMD_MASK           ;check if mask command
        jne     tmpsca4
        jmps    tmpsca2
tmpsca1 or      ah, ah                  ;check if extended key
        jnz     tmpsca4
        cmp     al, _KEY_CTL_V          ;check if ^V (mask command)
        jne     tmpsca3
tmpsca2 call    Read_Ctrl               ;read control character
        jc      tmpsca4
tmpsca3 mov     bx, tmpptr              ;load template pointer
        or      bx, bx                  ;check if template defined
        jnz     tmpsca6
tmpsca4 pop     ax
        stc
        ret
tmpsca5 inc     bx                      ;next character
tmpsca6 cmp     BYTE [bx], 0            ;quit if end of template
        je      tmpsca4
        cmp     [bx], al                ;check if matching character
        jne     tmpsca5
        pop     ax
        clc
        ret
        ENDP

;========================================================================
; Insert up to character.

Temp_Ins PROC   NEAR
        call    Temp_Scan               ;scan for character
        jc      tmpins3
        mov     si, tmpptr              ;load template pointer
        jmps    tmpins2
tmpins1 mov     ah, [si]                ;next template character
        call    Ins_Char                ;insert character
        inc     si                      ;increment pointer
tmpins2 cmp     si, bx                  ;check if finished
        jne     tmpins1
        mov     tmpptr, bx              ;save pointer
        or      al, UPDLIN              ;redisplay line
tmpins3 ret
        ENDP

;========================================================================
; Skip up to character.

Temp_Skip PROC  NEAR
        call    Temp_Scan               ;scan for character
        jc      tmpskp1
        mov     tmpptr, bx              ;save new pointer
tmpskp1 ret
        ENDP

;========================================================================
; Store a character to the current location.
;
; In: AH= character.

Ins_Char PROC   NEAR
        cmp     di, dx                  ;check if at end
        je      inschr1
        test    flags, INSERT OR SECINS ;check if insert
        jz      inschr3
inschr1 cmp     ch, cl                  ;check if buffer full
        je      inschr4

        push    cx
        push    di
        push    si
        mov     cx, dx
        sub     cx, di                  ;bytes to move
        jz      inschr2
        mov     di, dx                  ;destination is end
        mov     si, di
        dec     si                      ;source is last byte
        std
        rep
        movsb                           ;shift bytes to right
        cld
        or      al, UPDRIG              ;update to right
inschr2 pop     si
        pop     di
        pop     cx

        inc     cl                      ;increment bytes in buffer
        inc     dx                      ;increment end pointer

inschr3 mov     [di], ah                ;store byte
        inc     di                      ;advance pointer
        or      al, UPDOLD OR RESET OR HRESET ;flags

inschr4 ret
        ENDP

;========================================================================
; Delete the character at the current location.

Del_Char PROC   NEAR
        cmp     di, dx                  ;check if at end
        je      delchr1
        dec     cl                      ;reduce bytes in buffer
        dec     dx                      ;decrement end pointer

        push    cx
        push    di
        mov     si, di                  ;current location
        inc     si                      ;start with next byte
        mov     cx, dx
        sub     cx, di                  ;bytes to move
        rep
        movsb                           ;copy bytes
        pop     di
        pop     cx

        or      al, UPDRIG OR RESET OR HRESET
        ret

delchr1 call    Temp_Next               ;skip template character
        ret
        ENDP

;========================================================================
; Insert a string.
;
; In: SI= start of string.

        PROC   NEAR
strins1 mov     ah, al
        call    Ins_Char                ;insert character
Ins_Str LABEL   NEAR                    ;entry point
        lodsb                           ;load next character
        or      al, al                  ;check if NUL
        jnz     strins1
        ret
        ENDP

;========================================================================
; Load older or newer history line.

        PROC    NEAR

;--- load newest or oldest line

His_Oldest LABEL NEAR
        xchg    bx, si                  ;swap edges
His_Newest LABEL NEAR
        mov     curhis, 0               ;zero entry
        jmps    hisloa1

;--- load older line

His_Older LABEL NEAR
        test    ah, HRESET              ;check if flag set
        jnz     hisold2
hisold1 mov     ax, OFFSET Str_Prev     ;routine
        jmps    hisloa1
hisold2 mov     ax, curhis              ;load history
        or      ax, ax                  ;check if zero
        jz      hisold1                 ;ignore flag if so
hisold3 push    di
        mov     si, ax                  ;reload current line
        jmps    hisloa2

;--- load trace

His_Trace LABEL NEAR
        test    ah, HRESET              ;check if flag set
        jnz     histra2
        cmp     lasthis, 0              ;check if saved entry
        je      histra2
        cmp     curhis, 0               ;check current entry
        jne     histra2
        test    ah, PURHIS OR DELHIS    ;check if deleted
        jz      histra1                 ;must go down if not
        push    di
        mov     di, lasthis
        jmps    hisloa3
histra1 mov     ax, lasthis             ;copy saved to current
        mov     curhis, ax              ;
        jmps    hisnew1
histra2 test    ah, HTRACE              ;check if recover
        jz      hisloa6

;--- load newer line

His_Newer LABEL NEAR
        test    ah, HRESET              ;check if flag set
        jz      hisnew1
        mov     ax, curhis              ;load history
        or      ax, ax                  ;check if zero
        jz      hisnew1                 ;ignore flag if so
        jmps    hisold3
hisnew1 mov     ax, OFFSET Str_Next     ;routine
        xchg    bx, si                  ;swap edges

;--- load a new history line

hisloa1 push    di
        mov     di, hisbeg
        cmp     di, hisend
        je      hisloa5
        mov     di, curhis              ;load current location
        or      di, di                  ;check if empty location
        jz      hisloa2
        cmp     di, bx                  ;check if at end
        je      hisloa4
        call    ax                      ;get new line
        mov     si, di

hisloa2 mov     di, si                  ;new line must be in DI
hisloa3 mov     al, RESET               ;flags
        jmps    His_Reload

;--- finished, wrapped around: goto empty line

hisloa4 pop     di
        sub     al, al                  ;zero flags
        call    Clear_Line              ;clear the line if no entries
        ret

;--- finished, unsuccessful search or no history: do nothing

hisloa5 pop     di
hisloa6 sub     al, al                  ;zero flags
        ret
        ENDP

;========================================================================
; Search for matching history line.

        PROC    NEAR

His_Reverse LABEL NEAR
        push    dx
        mov     si, OFFSET His_Rev      ;reverse search
        mov     dx, rfirst              ;first entry
        call    His_Search              ;perform search
        jc      hisrev0
        mov     rfirst, dx              ;save updated value
        mov     ffirst, ax              ;reset forward
        jmps    hisrev0

His_Forward LABEL NEAR
        push    dx
        mov     bx, si                  ;end is last
        mov     si, OFFSET His_Fwd      ;forward search
        mov     dx, ffirst              ;first entry
        call    His_Search              ;perform search
        jc      hisrev0
        mov     ffirst, dx              ;save updated value
        mov     rfirst, ax              ;reset reverse

hisrev0 pop     dx
        jc      hisloa6
        mov     di, ax                  ;return pointer
        sub     al, al                  ;zero flags
        push    ax                      ;dummy value
        ENDP

;*** no 'ret' in previous routine, drop down to His_Reload below ***

;========================================================================
; Load the selected command history entry. Entered from one of the
; history loading routines above.

His_Reload PROC NEAR
        pop     dx                      ;drop cursor location
        mov     curhis, di              ;save pointer
        push    ax
        push    cx
        call    Str_Length              ;load string length
        mov     al, inpmax              ;maximum bytes
        cmp     cl, al                  ;check if too long
        jbe     hisrel1
        mov     cl, al                  ;use maximum length
hisrel1 mov     al, cl                  ;save bytes in buffer
        mov     si, di                  ;source
        mov     di, bp                  ;copy to input buffer
        rep
        movsb                           ;copy bytes
        mov     dx, di                  ;reload end of buffer
        mov     scroll, cl              ;zero scroll
        pop     cx
        mov     cl, al                  ;bytes in buffer

        mov     al, cols                ;load columns
        cmp     cl, al                  ;check if short enough
        jb      hisrel2

        sub     ah, ah                  ;AX gets length
        dec     ax                      ;column offset
        add     ax, bp                  ;offset from begin
        mov     di, ax                  ;cursor location

hisrel2 pop     ax
        call    Temp_Reset              ;reset template
        or      al, UPDLIN              ;set flags
        and     flags2, NOT HRESET      ;clear flags
        ret
        ENDP

;========================================================================
; Search for matching history line.

His_Search PROC NEAR
        push    di

        cmp     bx, hisend              ;check if no history
        stc                             ;need carry if exit
        je      hissea4                 ;exit if so

;--- set search bytes

        cmp     search, 0               ;check if byte defined
        jnz     hissea1
        mov     search, 0FFH            ;set zero byte search
        or      cl, cl
        jz      hissea1
        mov     search, cl              ;use current bytes in buffer

;--- starting entry

hissea1 mov     di, curhis              ;get current entry
hissea2 call    si                      ;perform search
        jc      hissea4                 ;exit if quit

;--- check if wrapped to original entry

        mov     ax, dx                  ;load previous match
        cmp     ax, di                  ;check if wrapped to first found
        stc
        je      hissea4
        or      ax, ax                  ;check if defined
        jnz     hissea3
        mov     dx, di                  ;save

;--- check if identical to current buffer

hissea3 push    cx
        push    di
        push    si
        sub     ch, ch
        mov     si, bp
        call    Str_Comp                ;compare
        pop     si
        pop     di
        pop     cx
        je      hissea2                 ;loop back if equal
        clc

hissea4 mov     ax, di                  ;return entry
        pop     di
        ret
        ENDP

;========================================================================
; Reverse history search routines.

His_Rev PROC    NEAR
        push    si
        mov     si, bp                  ;start of string
        or      di, di                  ;check if start at beginning
        jz      hisrev1

;--- switch to previous entry

        cmp     di, bx                  ;check if at start
        jne     hisrev2
hisrev1 mov     di, hisend              ;load end
hisrev2 call    Str_Prev                ;previous entry

;--- perform search

        push    di
        mov     ax, OFFSET Str_Prev     ;switch routine
        call    Str_Search              ;perform search
        pop     ax
        jnc     hisrev3                 ;jump if found
        cmp     ax, hisend              ;check if searched from end
        stc
        je      hisrev3
        mov     di, hisend              ;move to end
        call    Str_Prev                ;previous entry
        mov     ax, OFFSET Str_Prev     ;switch routine
        call    Str_Search              ;search again
hisrev3 pop     si
        ret
        ENDP

;========================================================================
; Forward history search routines.

His_Fwd PROC    NEAR
        push    si
        mov     si, bp                  ;start of string
        or      di, di                  ;check if start at beginning
        jz      hisfwd1

;--- switch to next entry

        cmp     di, bx                  ;check if at start
        jne     hisfwd2
hisfwd1 mov     di, hisbeg              ;load beginning
        jmps    hisfwd3
hisfwd2 call    Str_Next                ;next string

;--- perform search

hisfwd3 push    di
        mov     ax, OFFSET Str_Next     ;switch routine
        call    Str_Search              ;perform search
        pop     ax
        jnc     hisfwd4                 ;jump if found
        cmp     ax, hisbeg              ;check if searched from beginning
        stc
        je      hisfwd4
        mov     di, hisbeg              ;move to beginning
        mov     ax, OFFSET Str_Next     ;switch routine
        call    Str_Search              ;search again
hisfwd4 pop     si
        ret
        ENDP

;========================================================================
; Add the current input to the history.

His_Add PROC    NEAR
        or      cl, cl                  ;check if zero bytes
        jz      hisadd1                 ;exit if so
        push    di
        mov     di, bp
        mov     bl, cl
        sub     bh, bh
        mov     BYTE [di + bx], 0       ;store NUL
        call    Heap_Add                ;add to heap
        pop     di
hisadd1 ret
        ENDP

;========================================================================
; Delete the current history item.

His_Delete PROC NEAR
        mov     histmp, OFFSET His_Older ;goto older entry
        call    His_Erase               ;erase
        jnc     hisdel1                 ;jump if erased
        call    Clear_Line              ;clear line if failure
hisdel1 ret
        ENDP

;========================================================================
; Remove current history item (for PURHIS and DELHIS option).

His_Remove PROC NEAR
        push    ax
        push    si
        mov     histmp, OFFSET hisrem1  ;dummy routine, don't do anything
        call    His_Erase               ;erase
        pop     si
        pop     ax
hisrem1 ret
        ENDP

;========================================================================
; Erase a history line.
;
; In: CX= routine to call if about to erase.
;
; Out: CY= set if no erased.

His_Erase PROC  NEAR
        cmp     curhis, 0               ;check if no current
        je      hisera1
        test    ah, HRESET              ;check if line changed
        jnz     hisera1
        push    curhis
        call    histmp                  ;call handler
        pop     si
        push    cx
        push    di
        mov     di, si
        call    Str_Next                ;goto end of string
        xchg    di, si                  ;set up source and destination
        mov     cx, hisend              ;end address
        sub     cx, si                  ;get bytes to move
        rep
        movsb                           ;shift table
        mov     hisend, di              ;new end of table
        pop     di
        pop     cx
        clc
        ret
hisera1 stc
        ret
        ENDP

;========================================================================
; Clear the command history.

His_Clear PROC  NEAR
        mov     curhis, 0               ;zero history entry
        mov     lasthis, 0              ;zero last history entry
        mov     bx, hisbeg
        mov     hisend, bx              ;set end of table
        call    Temp_Reset              ;reset template
        ret
        ENDP

;========================================================================
; Turn history on or off.

His_On  PROC    NEAR
        and     flags4, NOT HSKIP       ;clear flag
        ret
        ENDP

His_Off PROC    NEAR
        or      flags4, HSKIP           ;set flag
        ret
        ENDP

;========================================================================
; Set insert mode.

Set_Ins PROC    NEAR
        or      flags, INSERT           ;set flag
        jmps    setins1

Clear_Ins LABEL NEAR
        and     flags, NOT INSERT       ;clear flag
        jmps    setins1

Toggle_Ins LABEL NEAR
        xor     flags, INSERT           ;toggle flag

setins1 push    ax
        call    Cur_On                  ;reset cursor
        pop     ax
        ret
        ENDP

;************************************************************************
; Interpret Routines

;========================================================================
; Start a macro variable.

Src_Var PROC  NEAR

;--- load the variable name

        push    di
        mov     di, varbeg              ;start of variable area
        jmps    srcvar2
srcvar1 cmp     di, varend              ;check if variable too big
        je      srcvar2                 ;skip if so
        stosb                           ;store byte
srcvar2 call    _Src_Read               ;read character
        jz      srcvar3                 ;error if end of line
        cmp     al, varchr2             ;check if end of variable
        jne     srcvar1                 ;loop back if not

;--- search for the macro

        cmp     bh, varmax              ;check if nested too deep
        je      srcvar3                 ;error if so
        push    si
        mov     si, varbeg              ;start of variable
        sub     di, si                  ;calculate length
        mov     cx, di
        call    Mac_Find                ;search for variable
        pop     si
        jnc     srcvar3                 ;exit if not found
        push    bx
        mov     bx, varptr              ;nest stack pointer
        mov     [bx], si                ;save current location
        pop     bx
        add     varptr, 2               ;increment pointer
        call    Str_Next                ;skip macro name
        mov     si, di                  ;new source
        inc     bh                      ;increment stack level

srcvar3 pop     di
        ret
        ENDP

;========================================================================
; Load a byte.

Src_Load PROC   NEAR
        mov     varsav, si              ;save source location
        push    varptr                  ;save stack pointer
        pop     varvar
        mov     varlev, bh              ;save stack level

;--- load a byte

srcloa1 lodsb                           ;load a byte
        or      al, al                  ;check if zero
        jnz     srcloa2                 ;exit if non-zero
        test    bl, PARA1 OR PARA2      ;check if in parameter
        jnz     srcloa2                 ;exit if so
        or      bh, bh                  ;check if pushed source
        jnz     srcloa3                 ;jump if so
srcloa2 and     bl, NOT SUBST           ;clear substitute flag
        ret

;--- end of string, pop level

srcloa3 sub     varptr, 2               ;decrement pointer
        push    bx
        mov     bx, varptr              ;load pointer
        mov     si, [bx]                ;get previous source
        pop     bx
        dec     bh                      ;decrement level
        jmps    srcloa1
        ENDP

;========================================================================
; Put back a read byte.

Src_Unload PROC NEAR
        mov     si, varsav              ;reload source
        push    varvar                  ;reload stack pointer
        pop     varptr
        mov     bh, varlev              ;reload level
        ret
        ENDP

;========================================================================
; Save the source and zero the current quote.

Src_Save PROC   NEAR
        mov     parquo, ah              ;save quote
        mov     parptr, si              ;save pointer
        sub     ah, ah                  ;zero quote
        ret
        ENDP

;========================================================================
; Prepare to reread last character.

Src_Back PROC   NEAR
        mov     si, lastptr             ;source
        push    lastvar                 ;stack pointer
        pop     varptr
        mov     bh, lastlev             ;level
        mov     ah, lastquo             ;quote
        ret
        ENDP

;========================================================================
; Read a character.
;
; In: SI= source; DX= arguments; BL= flags.
;
; Out: AL= character; CY= set if delimiter (or end of line); ZF= set if
;      end of line; SI, DX, BL= updated.

Src_Read PROC   NEAR
srcrea1 mov     ah, quoflg              ;load quote flag
        call    _Src_Read               ;read a character
        mov     quoflg, ah              ;save quote flag
        lahf
        test    bl, SINK                ;check if sink in effect
        jnz     srcrea1                 ;loop back if so
        sahf
        ret

;=== start of secondary read routine

        PROC    NEAR

;--- delimiter

srcrea2 test    bl, RECURS              ;check if recursive
        jnz     srcrea3
        test    bl, PARA2               ;end of parameter
        jnz     srcrea4
srcrea3 or      al, al                  ;set ZF
        stc
        ret

;--- end of parameter

srcrea4 call    Src_Back                        ;last character
        cmp     si, partal                      ;check if past old tail
        jbe     srcrea5
        mov     partal, si                      ;save tail
srcrea5 mov     ah, parquo                      ;restore quote flag
        mov     si, parptr                      ;restore pointer
        and     bl,NOT (PARA1 OR PARA2 OR SINK) ;clear flags

;=== routine entry point

_Src_Read LABEL NEAR

srcrea6 mov     lastptr, si             ;save location
        push    varptr
        pop     lastvar
        mov     lastlev, bh             ;save location
        mov     lastquo, ah             ;save quote

        call    Src_Load                ;load byte

srcrea7 or      al, al                  ;check if end of line
        jz      srcreaA
        cmp     al, param               ;check if parameter
        je      srcreaG
        or      ah, ah                  ;check if within quotes
        jnz     srcreaD
        cmp     al, chain               ;check if chain
        je      srcreaA
        cmp     al, DELIM               ;check if delimiter
        je      srcrea2
        cmp     al, remove              ;check if removable quote
        je      srcreaE
        cmp     al, quote1              ;check if quote
        je      srcrea8
        cmp     al, quote2              ;check if quote
        jne     srcrea9

;--- non-delimiter

srcrea8 xor     ah, al                  ;toggle quote flag
srcrea9 or      al, al                  ;set ZF
        clc
        ret

;--- end of line

srcreaA test    bl, RECURS              ;check if recursive
        jnz     srcreaB
        test    bl, PARA1 OR PARA2      ;check if in parameter
        jnz     srcrea4
        or      dx, dx                  ;check if expanding macro
        jz      srcreaB
        test    bl, SRCEOL              ;check if already reached end of line
        jz      srcreaC

srcreaB sub     ah, ah                  ;set ZF and clear quote
        stc
        ret

;--- switch to tail

srcreaC call    Src_Back                ;reread last character
        or      bl, PARA1 OR SRCEOL     ;set flags
        test    flags4, ASINK           ;check if autosink
        jnz     srcreaF
        call    Src_Save                ;save source
        mov     si, partal              ;switch to tail
        jmps    srcrea6

;--- within quotes, check for closing quote

srcreaD cmp     al, ah                  ;check if closing quote
        jne     srcrea9
        cmp     al, quote1              ;check if standard quote one
        je      srcrea8
        cmp     al, quote2              ;check if standard quote two
        je      srcrea8
srcreaE xor     ah, al                          ;toggle
        test    bl, PARA1 OR PARA2 OR QUOTE     ;check flags
        jz      srcrea9                         ;return as non-delimiter
        jmp     srcrea6                         ;absorb

;--- sink parameter

srcreaF mov     al, PARA1 OR RECURS OR SINK     ;set flags
        sub     cl, cl                          ;parameter zero
        jmps    srcreaJ

;--- parameter

srcreaG test    bl, PARA1 OR PARA2 OR SUBST ;check if in parameter
        jnz     srcrea9
        or      dx, dx                  ;check if expanding macro
        jz      srcrea9

        call    Src_Load                ;load character
        call    To_Upper                ;convert to uppercase

        push    bx
        mov     bx, OFFSET subtab       ;substitution table
srcreaH cmp     al, [bx + 1]            ;check if matching character
        je      srcreaK
        inc     bx                      ;increment pointer
        inc     bx                      ;
        cmp     bx, OFFSET subend       ;check if end of table
        jne     srcreaH
        pop     bx

        test    bl, RUNCMD              ;check if loading run command
        jnz     srcreaM

        cmp     al, varchr1             ;check if variable
        je      srcreaL
        cmp     al, 'S'                 ;check if sink
        je      srcreaF

        mov     ch, al                  ;save character
        sub     cl, cl                  ;zero argument number

        cmp     al, allarg              ;check if all arguments
        je      srcreaI

        sub     al, '0'                 ;convert to digit
        cmp     al, 9                   ;check if digit
        ja      srcreaM
        mov     cl, al                  ;argument number

        call    Src_Load                ;load character
        mov     ch, al                  ;save character

        sub     al, '0'                 ;convert to digit
        cmp     al, 9                   ;check if digit
        ja      srcreaI

        push    ax
        mov     ah, 10                  ;base ten
        xchg    al, cl                  ;current value
        mul     al, ah                  ;times ten
        add     cl, al                  ;add value
        pop     ax

        call    Src_Load                ;load character
        mov     ch, al                  ;save character

srcreaI mov     al, PARA1 OR RECURS     ;all parameter flag
        cmp     ch, allarg              ;check if ended with all arguments
        je      srcreaJ
        mov     al, PARA2 OR RECURS     ;single parameter flag
        call    Src_Unload              ;reread last character

srcreaJ or      bl, al                  ;set flags
        call    Src_Save                ;save source
        mov     si, dx                  ;load argument pointer
        sub     ch, ch                  ;zero high byte
        jmps    srcreaO

;--- substitute character

srcreaK mov     al, [bx]                ;load character
        pop     bx
        or      al, al                  ;check if undefined
        jz      srcreaM
        or      bl, SUBST               ;substitute character loaded
        jmp     srcrea7

;--- macro variable

srcreaL call    Src_Var                 ;start reading variable
        jmp     srcrea6

;--- invalid parameter

srcreaM mov     al, param               ;reload parameter character
        call    Src_Unload              ;reread character
        jmp     srcrea9

;=== find numbered field

;--- skip field

srcreaN call    _Src_Read               ;read character
        jz      srcreaP                 ;exit if end of line
        jnc     srcreaN                 ;loop while non-delimiter

;--- skip delimiters

srcreaO call    _Src_Read               ;read character
        jz      srcreaP                 ;exit if end of line
        jc      srcreaO                 ;loop while delimiter
        jcxz    srcreaP                 ;exit if field zero
        loop    srcreaN                 ;loop if haven't reached count

srcreaP call    Src_Back                ;reread last character
        and     bl, NOT RECURS          ;clear flag
        jmp     srcrea6                 ;goto read next character
        ENDP

        ENDP

;========================================================================
; Reset all macro and source data.

Mac_Clear PROC  NEAR
        mov     ax, bufend              ;end of buffer
        mov     bufptr, ax              ;reset pointer
        mov     ax, keybeg              ;end of keyboard buffer
        mov     keyptr, ax              ;reset pointer
        mov     ax, tabbeg              ;start of macro table
        mov     tabptr, ax              ;reset pointer
        mov     ax, vartab              ;start of variable table
        mov     varptr, ax              ;reset pointer
        mov     flags3, 0               ;zero interpret flags
        mov     keyloo, MAXLOO          ;reset loop count
        call    His_On                  ;turn history on
        ret
        ENDP

;************************************************************************
; File Name Matching Routines

;========================================================================
; Reset name pointers.
;
; In: SI= pointer value.

Name_Reset PROC NEAR
        mov     nambeg, si
        mov     namtop, si
        mov     namptr, si
        ret
        ENDP

;========================================================================
; Save the last field to the file name area.

Name_Save PROC  NEAR
        push    cx
        push    di

;--- find length of field

        mov     cx, di
namsav1 cmp     di, bp                  ;check if beginning of line
        je      namsav2
        dec     di                      ;reverse
        call    Is_Delim                ;check if delimiter
        jc      namsav1                 ;loop back if not
        inc     di                      ;don't include delimiter
namsav2 sub     cx, di                  ;bytes to needed
        inc     cx                      ;include NUL

;--- check if fits in total save area

        mov     ax, namend              ;calculate total size
        sub     ax, nambeg              ;
        cmp     cx, ax                  ;check if string too large
        ja      namsav5

;--- delete older entries until it fits

        mov     si, di                  ;start of field
        mov     di, namtop              ;end of start area

namsav3 mov     ax, namend              ;calculate length of first entry
        sub     ax, di                  ;
        cmp     cx, ax                  ;check if fits
        jbe     namsav4

        push    cx
        push    si
        mov     cx, di
        mov     di, nambeg              ;start of area
        mov     si, di
        call    Str_Next                ;next string
        xchg    si, di
        sub     cx, si                  ;calculate bytes to move
        rep
        movsb                           ;delete first entry
        pop     si
        pop     cx
        jmps    namsav3

;--- copy to save area

namsav4 mov     namptr, di              ;save current pointer
        dec     cx                      ;no NUL
        rep
        movsb                           ;copy to save area
        sub     al, al                  ;NUL
        stosb                           ;store it
        mov     namtop, di              ;save new top

namsav5 pop     di
        pop     cx
        ret
        ENDP

;========================================================================
; Try to load previous file name.

Name_Prev PROC  NEAR
        test    ah, FBEGUN              ;check if possible saved names
        jz      nampre2                 ;exit if not
        mov     si, namptr              ;current pointer
        cmp     si, nambeg              ;check if empty
        je      nampre1
        push    di
        mov     di, si
        call    Str_Prev                ;get previous string
        mov     si, di
        mov     namptr, di              ;save pointer
        pop     di
        push    si
        call    Del_Prev                ;delete previous name
        pop     si
        call    Ins_Str                 ;replace on command line
nampre1 call    Name_Set                ;set return values
nampre2 ret
        ENDP

;========================================================================
; Try to load next name.
;
; In: BX= Append routine.
;
; Out: CY= set if loaded.

Name_Next PROC  NEAR
        test    ah, FBEGUN              ;check if possible saved names
        jz      namnex1
        mov     si, namptr              ;current pointer
        cmp     si, namtop              ;check if none available
        je      namnex1
        push    di
        mov     di, si
        call    Str_Next                ;next string
        mov     si, di
        pop     di
        cmp     si, namtop              ;check if none available
        je      namnex1
        mov     namptr, si              ;save pointer
        push    si
        call    bx                      ;append routine
        pop     si
        call    Ins_Str                 ;replace on command line
        call    Name_Set                ;set return values
        stc
namnex1 ret
        ENDP

;========================================================================
; Append routine.

Name_App PROC   NEAR
        mov     ah, ' '                 ;space
        call    Ins_Char                ;insert it
        ret
        ENDP

;========================================================================
; File name completion routines.

        PROC    NEAR

Name_Match LABEL NEAR
        mov     bx, OFFSET Del_Prev
        call    Name_Next
        jnc     nammat2
nammat1 jmp     nammatP

Name_Append LABEL NEAR
        mov     bx, OFFSET Name_App
        call    Name_Next
        jc      nammat1

nammat2 push    bx
        or      flags, SECINS           ;set secondary insert
        mov     al, ah
        test    al, FBEGUN              ;check if already started
        jz      nammat3
        mov     ah, 4FH                 ;search for next function
        test    al, FERROR              ;check if previous failed search
        jnz     nammat7
        jmp     nammatI

;--- allocate space

nammat3 call    Name_Reset              ;reset name data, value unimportant
        mov     ax, namsiz              ;load bytes needed  ; as long as all
        or      ax, ax                  ;check if none      ; pointers same
        jz      nammat7                 ;no matching
        push    cx
        mov     cx, ax                  ;bytes needed for name and record
        call    Heap_Rel                ;allocate memory
        pop     cx
        jc      nammat7                 ;jump if error
        mov     si, hisend              ;end of heap
        mov     nampat, si              ;name pattern
        add     si, MAXNAM + 1
        mov     namrec, si              ;name record
        add     si, SEAREC
        mov     BYTE [si], 0            ;need NUL at start of table
        inc     si
        call    Name_Reset              ;set pointers
        sub     ax, XHEAP               ;calculate save size
        add     si, ax
        mov     namend, si              ;save pointer
        jmps    nammat5

;--- position cursor to end of field if middle of field

nammat4 call    Cur_Right               ;move right
nammat5 cmp     di, dx                  ;check if end of line
        je      nammat6
        call    Is_Delim                ;check if delimiter
        jc      nammat4

nammat6 pop     ax
        mov     ax, OFFSET Del_Prev     ;always delete initial pattern
        push    ax

;------------------------------------------------------------------------
; Extract file name from input line.

;--- check if empty line or delimiter to left of cursor

        cmp     di, bp                  ;check if at start
        je      nammat8
        dec     di                      ;previous character
        call    Is_Delim                ;check if delimiter
        inc     di                      ;restore DI
        jnc     nammat8

;--- non-delimiter to left of cursor, use field as pattern

        push    di
        call    Cur_Prev                ;get start of word
        mov     bx, di
        mov     si, di
        pop     di

        sub     bx, di                  ;get length
        neg     bx                      ;adjust sign
        cmp     bx, MAXNAM              ;check if too long
        jbe     nammat9
nammat7 jmp     nammatQ

;=== copy filespec and determine name and extension

nammat8 pop     ax
        mov     ax, OFFSET nammatR      ;no field, just insert
        push    ax
        sub     bx, bx                  ;zero bytes in field

nammat9 push    cx
        push    dx
        push    di
        push    bp

        mov     cx, bx                  ;number of bytes
        mov     di, nampat              ;place to store
        mov     bp, di                  ;BP <- end of path
        sub     bx, bx                  ;BL/BH <- length of name / extension
        sub     dl, dl                  ;extension increment
        sub     ax, ax                  ;no character or drive if CX=0
        jcxz    nammatE                 ;skip copy if zero characters

;--- loop for each character

nammatA mov     dh, al                  ;save last letter
        lodsb                           ;load byte
        stosb                           ;store byte

        inc     bl                      ;update file name length
        add     bh, dl                  ;update extension length

        cmp     al, '\'                 ;check if end of path
        je      nammatB
        cmp     al, ':'                 ;check if end of drive
        jne     nammatC
        mov     ah, dh                  ;save drive letter
nammatB mov     bp, di                  ;save start of name
        sub     bx, bx                  ;zero name and extension length
        sub     dl, dl                  ;zero extension increment

nammatC cmp     al, '.'                 ;check if start of extension
        jne     nammatD
        mov     dl, 1                   ;set extension increment
nammatD loop    nammatA

nammatE mov     pthend, bp              ;save end of path
        mov     drive, ah               ;save drive

;--- add asterisks to name and/or extension

        mov     dh, al                  ;save the last character
        mov     al, '*'                 ;load asterisk
        or      dl, dl                  ;check if extension
        jz      nammatF

;--- extension exists

        cmp     bh, 3                   ;check extension length
        jae     nammatH
        cmp     dh, al                  ;check if asterisk
        je      nammatH
        stosb                           ;store asterisk
        jmps    nammatH

;--- no extension exists

nammatF cmp     bl, 8                   ;check name length
        jae     nammatG
        cmp     dh, al                  ;check if asterisk
        je      nammatG
        stosb                           ;store asterisk
nammatG mov     ax, '.*'                ;default extension
        stosw                           ;store it

;--- finished with name

nammatH sub     al, al                  ;nul
        stosb                           ;store it
        pop     bp
        pop     di
        pop     dx
        pop     cx

        mov     ah, 4EH                 ;search function

;------------------------------------------------------------------------
; Search for matching file name.
;
; In: AH= 4EH for first match; 4FH for next match.

nammatI push    cx
        push    dx
        push    ds
        push    es

        push    ax
        test    flags4, DOSV32          ;check if DOS v3.2 or higher
        jz      nammatJ
        mov     al, drive               ;load drive
        or      al, 20H                 ;convert to lowercase
        sub     al, 'a' - 1             ;convert to drive number
        jbe     nammatJ
        cmp     al, 26                  ;check upper range
        ja      nammatJ
        mov     bl, al
        mov     ax, 440FH               ;set drive function
        int     21H                     ;execute
nammatJ pop     cx

;--- save the current DTA on stack

        mov     ah, 2FH                 ;get DTA function
        int     21H                     ;execute
        push    es
        push    bx

;--- point the DTA to record

        mov     ah, 1AH                 ;set DTA function
        mov     dx, namrec              ;name record
        int     21H                     ;execute

;--- perform the search

        mov     ax, cx                  ;restore function
        mov     dx, nampat              ;name pattern     ;-- only used if
        mov     cx, namtyp              ;file types       ;   AH = 4EH
        int     21H                     ;execute

;--- restore the DTA

        pop     dx
        pop     ds
        lahf
        push    ax
        mov     ah, 1AH                 ;set DTA function
        int     21H                     ;execute
        pop     ax
        sahf

        pop     es
        pop     ds
        pop     dx
        pop     cx
        jc      nammatQ

;------------------------------------------------------------------------
; Copy file name to input line.

        pop     ax                      ;restore append or replace routine
        call    ax                      ;start append or replace

;--- copy path

        mov     si, nampat              ;load start of path
        jmps    nammatL
nammatK lodsb                           ;load byte
        call    namcas                  ;set case
        mov     ah, al
        call    Ins_Char                ;insert character
nammatL cmp     si, pthend              ;check if end of path
        jne     nammatK

;--- copy filename

        mov     si, namrec              ;load start of path
        push    si
        add     si, 30                  ;skip to name
        jmps    nammatN

nammatM call    namcas                  ;set case
        mov     ah, al
        call    Ins_Char                ;insert character
nammatN lodsb                           ;load byte
        or      al, al
        jnz     nammatM                 ;loop if non-zero

;--- append backslash

        pop     si
        test    flags4, SLASHED         ;check if add backslash
        jz      nammatO
        test    BYTE [si + 21], 10H     ;check if directory
        jz      nammatO
        mov     ah, '\'                 ;back slash
        call    Ins_Char                ;insert

nammatO call    Name_Save               ;save file name

Name_Set LABEL  NEAR                    ;routine entry point
nammatP and     flags2, NOT FERROR      ;clear flag
        mov     al, UPDLIN OR RESET OR HRESET OR FBEGUN  ;set flags
        ret

;--- invalid filespec or file not found

nammatQ or      flags2, FERROR          ;set flag
        mov     al, FBEGUN              ;flags
        pop     bx
nammatR ret
        ENDP

;------------------------------------------------------------------------
; NUL byte, beginning of heap.

        DB      0               ;NUL byte

RES_END                         ;end of resident code

;************************************************************************
; Install

Install

        UNUSED-
        INCLUDE 'c:\src\wasm\case1'
        INCLUDE 'c:\src\wasm\case2'
        INCLUDE 'c:\src\wasm\convert'
        INCLUDE 'c:\src\wasm\draw1'
        INCLUDE 'c:\src\wasm\enviro'
        INCLUDE 'c:\src\wasm\file'
        INCLUDE 'c:\src\wasm\intr'
        INCLUDE 'c:\src\wasm\intr2'
        INCLUDE 'c:\src\wasm\justify1'
        INCLUDE 'c:\src\wasm\keybrd'
        INCLUDE 'c:\src\wasm\memory'
        INCLUDE 'c:\src\wasm\message1'
        INCLUDE 'c:\src\wasm\mulplx2'
        INCLUDE 'c:\src\wasm\string'
        INCLUDE 'c:\src\wasm\video1'
        INCLUDE 'c:\src\wasm\video3'
        INCLUDE 'c:\src\wasm\video4'
        INCLUDE 'c:\src\wasm\video5'
        INCLUDE 'c:\src\wasm\video6'
        UNUSED+

;------------------------------------------------------------------------
; Setup.

;--- reduce memory allocation

        mov     bx, sp                  ;end of segment
        mov     cl, 4                   ;bits to shift
        shr     bx, cl                  ;convert to paragraphs
        test    sp, 1111B               ;check if odd bits
        jz      instal1
        inc     bx                      ;round up
instal1 mov     ah, 4AH                 ;modify memory allocation
        int     21H                     ;execute

;--- assign variable addresses

        mov     bp, OFFSET messR
        mov     ax, sp
        sub     ax, (MAXARG * 4) + 1    ;argument table, +1 for end NUL
        mov     argtab, ax              ;
        sub     ax, (MAXARG * 4) + 1    ;argument table 2
        mov     argtab2, ax             ;
        sub     ax, 81                  ;program name
        mov     pname, ax               ;
        sub     ax, 81                  ;configuration name
        mov     cname, ax               ;
        sub     ax, MAXLIN + 1          ;line buffer
        mov     linbuf, ax              ;
        sub     ax, MAXLIN + 2          ;key conversion buffer, allow for 2
        mov     keybuf, ax              ;    NUL's
        cmp     ax, OFFSET TRAN_END + EXTRA ;check pointer value
        jb      instal2
        mov     sp, ax                  ;set new stack below data

;--- define local heap

        mov     ax, OFFSET TRAN_END     ;end of data
        mov     _hisbeg, ax             ;history pointers
        mov     _hisend, ax
        mov     dx, ax
        mov     ax, sp
        sub     ax, dx                  ;raw bytes available
        sub     ax, EXTRA               ;space for stack and loader
        mov     heapmax, ax             ;maximum heap
        add     ax, dx
        mov     _macbeg, ax             ;macro pointers
        mov     _macend, ax

;--- relocate macro tail

        mov     cx, mtail               ;bytes in macro tail
        jcxz    instal4
        cmp     cx, heapmax             ;check space
        ja      instal4                 ;skip macros if not enough space

        mov     si, OFFSET TRAN_END     ;start of macros
        add     si, cx
        dec     si                      ;end of tail
        mov     di, _macbeg             ;beginning of macros
        dec     di                      ;starting location
        std
        rep
        movsb                           ;copy tail
        inc     di                      ;point to first byte
        mov     _macbeg, di             ;save new beginning
        jmps    instal4

;--- relay points

instal2 jmp     instal9
instal3 jmp     instalC

;--- copy resident heap and data

instal4 mov     bp, OFFSET messG
        mov     ax, multi               ;multiplex number
        call    PlxQue                  ;resident query
        jc      instal2
        or      dx, dx                  ;check if resident
        jz      instal5
        mov     rseg, dx                ;save segment

        mov     bp, OFFSET messR
        push    ds
        mov     ds, dx
        mov     ax, macend              ;calculate heap size
        sub     ax, hisbeg              ;
        pop     ds
        cmp     ax, heapmax             ;check if enough space
        ja      instal2

        push    ds
        mov     ds, dx
        mov     si, hisbeg
        seg     cs
        mov     di, _hisbeg
        mov     cx, ax
        cld
        rep
        movsb                           ;copy heap
        mov     si, OFFSET REL_TAR
        mov     di, OFFSET REL_BEG
        mov     cx, REL_SIZE
        rep
        movsb                           ;copy data
        pop     ds
        mov     ax, OFFSET TRAN_END - OFFSET RES_END
        add     _hisbeg, ax
        add     _hisend, ax
        add     _macbeg, ax
        add     _macend, ax
        cmp     _curhis, 0
        je      instal5
        add     _curhis, ax

;--- read switches

instal5 mov     di, OFFSET mess1        ;default exit message
        mov     bp, OFFSET mess2        ;default error message
        call    Configure               ;read switches
        jc      instal9

        mov     ax, rseg
        or      ax, ax
        jz      instal3

;------------------------------------------------------------------------
; Already resident, just pass switches.

        push    di
        push    es
        mov     es, ax
        mov     si, _hisbeg
        seg     es
        mov     di, hisbeg
        mov     cx, _macend
        sub     cx, _hisbeg
        cld
        rep
        movsb                                           ;copy heap
        mov     ax, OFFSET TRAN_END - OFFSET RES_END
        sub     _hisbeg, ax
        sub     _hisend, ax
        sub     _macbeg, ax
        sub     _macend, ax
        cmp     _curhis, 0
        je      instal6
        sub     _curhis, ax
instal6 mov     si, OFFSET REL_BEG
        mov     di, OFFSET REL_TAR
        mov     cx, REL_SIZE
        cld
        rep
        movsb                                           ;copy data
        pop     es
        pop     di

;------------------------------------------------------------------------
; Terminate.

;--- finished, no error

instal7 call    Show_Banner             ;display banner
        test    flags5, QUIET           ;check if quiet mode
        jnz     instal8
        or      di, di                  ;check if no message
        jz      instal8
        mov     ax, di
        call    MesPutL                 ;display message
instal8 mov     ax, 4C00H               ;terminate function
        int     21H                     ;execute

;=== error

instal9 call    Show_Banner             ;display banner
        mov     ax, bp
        call    MesPut                  ;display error message

        mov     ax, serror              ;load error switch
        cmp     al, ' '                 ;check if delimiter
        ja      instalA
        cmp     ah, ' '                 ;check if delimiter
        jbe     instalB
instalA push    ax
        mov     ax, OFFSET error1       ;error switch
        call    MesPut
        pop     ax
        call    Display_Char
        xchg    al, ah
        call    Display_Char
        mov     ax, OFFSET error2       ;error switch
        call    MesPut

instalB mov     ax, OFFSET crlf         ;blank line
        call    MesPut                  ;display
        mov     ax, 4CFFH               ;terminate function
        int     21H                     ;execute

;------------------------------------------------------------------------
; Not resident, install.

instalC test    sflags, QUI             ;check if just quit
        jnz     instal7                 ;jump if so

        mov     bp, OFFSET messF
        mov     ax, heap_h              ;history size
        add     ax, heap_m              ;add macro size
        jc      instal9
        add     ax, _namsiz
        jc      instal9
        cmp     ax, heapmax
        ja      instal9
        mov     heapsiz, ax             ;save heap size
        mov     ax, _macend             ;end of macros
        sub     ax, _hisbeg             ;calculate current size
        sub     ax, heapsiz             ;bytes difference
        jb      instal9
        call    Heap_Shrink             ;shrink heap
        jc      instal9

;--- adjust table addresses

        mov     ax, OFFSET TRAN_END - OFFSET RES_END
        sub     _hisbeg, ax
        sub     _hisend, ax
        sub     _macbeg, ax
        sub     _macend, ax

        mov     bp, OFFSET messF
        mov     ax, OFFSET RES_END      ;end of data
        add     ax, heapsiz             ;skip heap
        mov     tabbeg, ax              ;macro table beginning

        mov     dx, _macnes             ;maximum nest
        inc     dx                      ;one by default
        shl     dx
        shl     dx
        add     ax, dx
        jc      instalD
        mov     tabend, ax              ;table end
        sub     tabend, 4               ;point to last entry, not end
        mov     varbeg, ax              ;start of variable name buffer
        add     ax, _varlen
        mov     varend, ax              ;end of variable name buffer
        mov     vartab, ax              ;variable table
        mov     dl, _varmax             ;max variable levels
        sub     dh, dh
        add     dx, dx                  ;two bytes per entry
        add     ax, dx
        mov     bufbeg, ax              ;input buffer
        add     ax, _inpsiz
        jc      instalD
        mov     bufend, ax
        inc     ax                      ;NUL at end of buffer
        mov     keybeg, ax              ;key code buffer
        add     ax, _keysiz
        jc      instalD
        mov     keyend, ax
        jz      instalD
        mov     dx, ax
        sub     dx, OFFSET RES_END
        mov     WORD wdata2 + 10, dx    ;save size of private data
        push    ax                      ;save end location on stack
        add     ax, EXTRA               ;extra space for loader and stack
        jc      instalD
        mov     bp, OFFSET messE
        cmp     ax, sp
        jbe     instalE
instalD jmp     instal9

;--- copy relocate data

instalE mov     si, OFFSET REL_BEG      ;start start of data
        mov     di, OFFSET REL_TAR      ;relocation address
        mov     cx, REL_SIZE            ;bytes
        cld
        rep
        movsb                           ;copy table

;--- initialize data

        mov     oldloc, 0               ;zero old location
        mov     scroll+1, 0             ;zero high byte of scroll
        mov     search+1, 0             ;zero high byte of search

        mov     WORD wdata1 + 16, cs
        mov     WORD wdata2 + 2, cs
        mov     WORD wdata2 + 8, cs

        and     flags4, NOT DOSV32      ;clear version 3 flag
        mov     ah, 30H                 ;get DOS function
        int     21H                     ;execute
        xchg    al, ah                  ;swap order
        cmp     ax, 0302H               ;check version number
        jb      instalF
        or      flags4, DOSV32          ;set flag

;--- hook interrupts

instalF mov     al, 1BH
        mov     bx, OFFSET old1B
        mov     cx, OFFSET New1B
        call    IntChn                  ;chain interrupt 1B

        mov     al, 21H
        mov     bx, OFFSET old21
        mov     cx, OFFSET New21
        call    IntChn                  ;chain interrupt 21

        mov     ax, multi
        mov     cx, VER_LO*256+VER_HI
        mov     dx, OFFSET New2F
        call    PlxChn                  ;chain multiplex interrupt

;--- shift loader and jump to it

        call    Show_Banner             ;display banner
        test    flags5, QUIET           ;check if quiet mode
        jnz     instalG
        mov     ax, OFFSET mess4
        call    MesPutL                 ;display

instalG mov     si, OFFSET LOADER       ;start of loader
        mov     di, OFFSET TRAN_END
        push    di
        add     di, heapsiz             ;place to put loader
        push    di                      ;save for branch
        mov     cx, LOAD_SIZ            ;bytes to move
        cld
        rep
        movsb                           ;move loader
        mov     ax, OFFSET Mac_Clear    ;save routine address
        mov     cx, cmdlen              ;load command length
        retn                            ;tranfer to loader

;--- relocate heap and terminate, this code is moved out of the way

LOADER
        call    ax                      ;reset macro data

        mov     bx, tabbeg
        mov     WORD [bx+2], 0          ;store NULL pointer to top entry
        mov     di, bufend
        mov     BYTE [di], 0            ;store NUL to end of input buffer
        inc     di
        mov     [bx], di                ;store scope pointer to top entry

        jcxz    instalH
        mov     di, bufend              ;end of buffer
        sub     di, cx                  ;adjust pointer
        mov     bufptr, di              ;save start of command
        mov     si, OFFSET cmdbuf       ;start of command
        cld
        rep
        movsb                           ;copy
        or      BYTE flags3, RUNCMD     ;set flag

instalH pop     si
        mov     di, OFFSET RES_END      ;new location
        mov     cx, heapsiz
        cld
        rep
        movsb                           ;copy to new location
        pop     ax                      ;restore end offset
        Keep    ax                      ;terminate

cmdbuf  DS      MAXLIN + 1              ;command buffer

LOAD_SIZ EQU    $ - OFFSET LOADER       ;size of loader

EXTRA   EQU     LOAD_SIZ + 1000         ;loader size plus stack

;************************************************************************
; Switch Routines

        PROC    NEAR

;--- uninstall toddy

Option22
        mov     al, 1BH
        mov     cx, OFFSET New1B
        mov     dx, rseg
        call    IntChk                  ;check interrupt 1B
        jne     op22a
        mov     al, 21H
        mov     cx, OFFSET New21
        mov     dx, rseg
        call    IntChk                  ;check interrupt 21
        jne     op22a
        mov     ax, multi
        call    PlxChk                  ;check multiplex interrupt
        jne     op22a
        push    ds
        mov     ds, rseg
        mov     al, 1BH
        mov     bx, OFFSET old1B
        call    IntRel                  ;release interrupt 1B
        mov     al, 21H
        mov     bx, OFFSET old21
        call    IntRel                  ;release interrupt 21
        pop     ds
        mov     ax, multi
        call    PlxRel                  ;release multiplex interrupt
        mov     ax, rseg                ;load segment
        call    MemRel                  ;release memory
        mov     di, OFFSET mess5        ;exit message
        clc
        ret
op22a   mov     bp, OFFSET mess9        ;message
        stc
        ret

;--- enable toddy

Option20
        or      _flags, ENABLE          ;set flag
        mov     di, OFFSET messM        ;exit message
        call    Read_Null               ;check args
        ret

;--- disable toddy

Option21
        and     _flags, NOT ENABLE      ;set flag
        mov     di, OFFSET messN        ;exit message
        call    Read_Null               ;check args
        ret

;--- run a command

Option57
        push    di
        call    Read_StrX               ;read string
        jc      op57a
        mov     di, bx
        call    Str_Length              ;get length of symbol
        mov     si, di                  ;source
        mov     ax, rseg                ;load resident segment
        or      ax, ax                  ;check if resident
        jnz     op57b
        mov     cmdlen, cx              ;save command length
        mov     di, OFFSET cmdbuf       ;command buffer
        cld
        rep
        movsb                           ;copy to command buffer
        clc
op57a   pop     di
        ret
op57b   push    es
        push    ds
        mov     es, ax                  ;load segment
        mov     ds, ax
        call    Mac_Clear               ;reset macro data
        mov     di, bufend              ;end of buffer
        sub     di, cx                  ;adjust pointer
        mov     bufptr, di              ;save pointer
        or      BYTE flags3, RUNCMD     ;set flag
        pop     ds
        cld
        rep
        movsb                           ;copy command to buffer
        mov     _loaded, 0              ;reset preloaded command
        pop     es
        pop     di
        clc
        ret

;--- write settings

Option23
        call    Read_Null               ;check switch
        jc      op23a
        call    Write_Opts              ;write options
op23a   ret

;--- load switches

Option88
        call    Read_Name               ;retrieve name
        jnc     opt88a
        mov     ax, cname               ;default name
opt88a  call    Load_Config             ;load configuration
        mov     di, OFFSET messGG       ;exit message
        ret

;--- save switches

Option89
        call    Read_Name               ;retrieve name
        jnc     opt89a
        mov     ax, cname               ;default name
opt89a  call    Save_Config             ;save configuration
        mov     di, OFFSET messHH       ;exit message
op49a   ret                             ;borrow this RET from below

;--- display memory allocations

Option49
        call    Read_Null               ;check switch
        jc      op49a

        push    cx
        push    si

        mov     ax, OFFSET crlf         ;blank line
        call    MesPut                  ;display
        mov     ax, heap_h              ;history allocation
        mov     bx, OFFSET alloc1
        mov     cx, OFFSET allocD
        call    Display_Num2            ;display
        mov     ax, heap_m              ;macro allocation
        mov     bx, OFFSET alloc2
        mov     cx, OFFSET allocE
        call    Display_Num2            ;display
        mov     ax, _namsiz             ;extra allocation
        mov     bx, OFFSET alloc3
        mov     cx, OFFSET alloc3
        call    Display_Num2            ;display

        mov     ax, _macend             ;calculate total memory
        sub     ax, _hisbeg             ;
        mov     bx, OFFSET alloc4
        mov     cx, OFFSET alloc4
        call    Display_Num2            ;display
        mov     ax, _macend             ;calculate bytes used by macros
        sub     ax, _macbeg             ;
        mov     bx, OFFSET alloc5
        mov     cx, OFFSET alloc5
        call    Display_Num2            ;display
        mov     ax, _macbeg             ;calculate bytes available
        sub     ax, _hisbeg             ;
        mov     bx, OFFSET alloc6
        mov     cx, OFFSET allocF
        call    Display_Num2            ;display

        mov     ax, _macnes             ;macro nest
        push    ax
        mov     bx, OFFSET alloc7
        mov     cx, OFFSET allocC
        call    Display_Num1            ;display
        pop     ax
        shl     ax
        shl     ax
        push    ax
        sub     dx, dx                  ;zero high word
        mov     cx, 10                  ;base 10
        mov     bx, OFFSET numbuf       ;number buffer
        push    bx
        call    Num2Str                 ;convert to string
        pop     ax
        call    MesPut                  ;display number
        mov     ax, OFFSET alloc8
        call    MesPutL                 ;display end of message

        mov     al, _varmax             ;nest levels
        sub     ah, ah                  ;
        shl     ax                      ;bytes used by nest
        add     ax, _varlen             ;add bytes for name
        push    ax
        mov     bx, OFFSET allocG
        mov     cx, OFFSET allocG
        call    Display_Num2            ;display

        mov     ax, _keysiz             ;keyboard buffer size
        mov     bx, OFFSET alloc9
        mov     cx, OFFSET alloc9
        call    Display_Num2            ;display

        mov     ax, _inpsiz             ;input buffer size
        mov     bx, OFFSET allocA
        mov     cx, OFFSET allocA
        call    Display_Num2            ;display

        pop     ax                      ;restore variable bytes
        pop     dx                      ;restore macro nest bytes
        add     ax, dx
        add     ax, heap_m              ;add macro size
        add     ax, heap_h              ;add heap size
        add     ax, _namsiz             ;add extra size
        add     ax, _keysiz             ;add keyboard size
        add     ax, _inpsiz             ;add input size
        mov     bx, OFFSET allocB
        mov     cx, OFFSET allocB
        call    Display_Num2            ;display

        sub     di, di                  ;no exit message (and clear CY)
        pop     si
        pop     cx
        ret

;--- help

Option27
        push    si
        call    Read_Null               ;check switch
        jc      op27d
        push    cx
        sub     di, di                  ;zero once through flag
        mov     si, OFFSET prompt1      ;initial prompt
op27a   mov     ax, OFFSET helpt        ;display top part
        call    MesPutL                 ;
        mov     ax, OFFSET help1        ;display help page
        call    MesPutL                 ;
        call    op27e
        cmp     al, 'P'                 ;check if next page
        je      op27b
        or      di, di                  ;check if already came through
        jnz     op27c                   ;
        cmp     al, KEY_ESC             ;first time through, check if quit
        je      op27c
        cmp     al, KEY_CTL_C           ;first time through, check if quit
        je      op27c
op27b   or      di, 1                   ;set flag for once through
        mov     si, OFFSET prompt2      ;second prompt
        mov     ax, OFFSET helpt        ;display top part
        call    MesPutL                 ;
        mov     ax, OFFSET help2        ;display help page
        call    MesPutL                 ;
        call    op27e
        cmp     ax, 'P'                 ;check if other page
        je      op27a
op27c   pop     cx
        sub     di, di                  ;no return message (and zero CY)
op27d   pop     si
        ret
op27e   mov     ax, si                  ;display prompt
        call    MesPut                  ;
        call    KeyClr                  ;clear keyboard
        call    KeyWai                  ;wait for key
        push    ax
        mov     ax, OFFSET prompt3      ;clear prompt
        call    MesPut                  ;
        pop     ax
        call    ChrUpr                  ;convert key to upper case
        ret

;--- set quiet mode

Option39
        or      flags5, QUIET           ;set flag
        call    Read_Null               ;check switch
        ret

;--- set verbose mode

Option40
        and     flags5, NOT QUIET       ;clear flag
        call    Read_Null               ;check switch
        ret

;--- input buffer / workspace size

Option55
        mov     ax, 10                  ;base 10
        call    Read_Word               ;read number
        jc      op55b
        cmp     ax, MAXINP              ;minimum size
        jae     op55a
        mov     ax, MAXINP              ;set to minimum
op55a   mov     _inpsiz, ax             ;save it
        clc
op55b   ret

;--- define multiplex number

Option32
        mov     ax, 10                  ;base 10
        call    Read_Byte               ;read number
        mov     ah, al                  ;multiplex number
        mov     al, MPLX2               ;multiplex subfunction
        mov     multi, ax               ;save it
        ret

;--- buffer size

Option34
        mov     ax, 10                  ;base 10
        call    Read_Byte               ;read number
        mov     _sizchk, al             ;save it
        ret

;--- uppercase significant

Option47
        or      _flags, CASE            ;set flag
        call    Read_Null               ;check switch
        ret

;--- uppercase ignore

Option48
        and     _flags, NOT CASE        ;clear flag
        call    Read_Null               ;check switch
        ret

;--- public Windows session

Option86
        and     _flags4, NOT PRIVATE    ;clear flag
        call    Read_Null               ;check switch
        ret

;--- private Windows session

Option87
        or      _flags4, PRIVATE        ;set flag
        call    Read_Null               ;check switch
        ret

;--- preserve insert

Option24
        and     _flags, NOT IRESET      ;clear flag
        call    Read_Null               ;check switch
        ret

;--- reset insert

Option10
        or      _flags, IRESET          ;set flag
        call    Read_Null               ;check switch
        ret

;--- insert mode

Option08
        or      _flags, INSERT OR IVALUE        ;set flags
        call    Read_Null               ;check switch
        ret

;--- overwrite mode

Option09
        and     _flags, NOT (INSERT OR IVALUE)  ;clear flags
        call    Read_Null               ;check switch
        ret

;--- key command begin character

Option80
        mov     ax, 10                  ;base 10
        call    Read_Byte               ;read number
        mov     field1, al              ;save it
        ret

;--- key command end character

Option81
        mov     ax, 10                  ;base 10
        call    Read_Byte               ;read number
        mov     field2, al              ;save it
        ret

;--- literal character

Option05
        mov     ax, 10                  ;base 10
        call    Read_Byte               ;read number
        mov     _liter, al              ;save it
        ret

;--- parameter character

Option06
        mov     ax, 10                  ;base 10
        call    Read_Byte               ;read number
        mov     _param, al              ;save it
        mov     _param+1, al            ;save substitution copy
        ret

;--- chain character

Option07
        mov     ax, 10                  ;base 10
        call    Read_Byte               ;read number
        mov     _chain, al              ;save it
        ret

;--- quote character

Option50
        mov     ax, 10                  ;base 10
        call    Read_Byte               ;read number
        mov     _remove, al             ;save it
        ret

;--- text quote character one

Option51
        mov     ax, 10                  ;base 10
        call    Read_Byte               ;read number
        mov     _quote1, al             ;save it
        ret

;--- text quote character two

Option52
        mov     ax, 10                  ;base 10
        call    Read_Byte               ;read number
        mov     _quote2, al             ;save it
        ret

;--- multiple argument character

Option56
        mov     ax, 10                  ;base 10
        call    Read_Byte               ;read number
        mov     _allarg, al             ;save it
        ret

;--- start of variable

Option73
        mov     ax, 10                  ;base 10
        call    Read_Byte               ;read number
        mov     _varchr1, al            ;save it
        ret

;--- end of variable

Option74
        mov     ax, 10                  ;base 10
        call    Read_Byte               ;read number
        mov     _varchr2, al            ;save it
        ret

;--- text attribute

Option11
        mov     ax, 16                  ;base
        call    Read_Byte               ;read number
        mov     _atext, al              ;save it
        ret

;--- cursor attribute

Option12
        mov     ax, 16                  ;base
        call    Read_Byte               ;read number
        mov     _acurs, al              ;save it
        ret

;--- end attribute

Option13
        mov     ax, 16                  ;base
        call    Read_Byte               ;read number
        mov     _aend, al               ;save it
        ret

;--- select border attribute

Option61
        mov     ax, 16                  ;base
        call    Read_Byte               ;read number
        mov     winatr1, al             ;save it
        ret

;--- select selected attribute

Option62
        mov     ax, 16                  ;base
        call    Read_Byte               ;read number
        mov     winatr2, al             ;save it
        ret

;--- select text attribute

Option63
        mov     ax, 16                  ;base
        call    Read_Byte               ;read number
        mov     winatr3, al             ;save it
        ret

;--- define entry

Option16
        mov     ax, 16                  ;base
        call    Read_Word               ;read number
        mov     _cins, ax               ;save it
        mov     _cover, ax              ;save it
op16a   jc      op16c
        cmp     ax, DCURSOR             ;check if default
        je      op16b
        or      _flags, CURSOR          ;set flag
        call    Default_Cur             ;set default cursor
op16b   clc
op16c   ret

;--- define insert

Option17
        mov     ax, 16                  ;base
        call    Read_Word               ;read number
        mov     _cins, ax               ;save it
        jmps    op16a

;--- define overwrite

Option18
        mov     ax, 16                  ;base
        call    Read_Word               ;read number
        mov     _cover, ax              ;save it
        jmps    op16a

;--- define exit

Option19
        mov     ax, 16                  ;base
        call    Read_Word               ;read number
        mov     _cexit, ax              ;save it
        jmps    op16a

;--- custom cursor

Option14
        or      _flags, CURSOR          ;set flag
        call    Default_Cur             ;set default cursor
        call    Read_Null               ;check switch
        ret

;--- system cursor

Option15
        and     _flags, NOT CURSOR      ;clear flag
        call    Read_Null               ;check switch
        ret

;--- history size

Option03
        mov     ax, 10                  ;base 10
        call    Read_Word               ;read number
        mov     heap_h, ax              ;save it
        ret

;--- minimum length

Option01
        mov     ax, 10                  ;base 10
        call    Read_Byte               ;read number
        mov     _minlen, al             ;save it
        ret

;--- pure history

Option90
        call    His_Purify              ;purify history
        or      _flags2, PURHIS         ;set flag
        mov     _curhis, 0              ;zero current
        call    Read_Null               ;check switch
        ret

;--- impure history

Option91
        and     _flags2, NOT PURHIS     ;clear flag
        mov     _curhis, 0              ;zero current
        call    Read_Null               ;check switch
        ret

;--- delete selected history

Option66
        or      _flags2, DELHIS         ;set flag
        mov     _curhis, 0              ;zero current
        call    Read_Null               ;check switch
        ret

;--- preserve selected history

Option67
        and     _flags2, NOT DELHIS     ;clear flag
        mov     _curhis, 0              ;zero current
        call    Read_Null               ;check switch
        ret

;--- save unique input

Option02
        or      _flags2, UNIQUE         ;set flag
        call    Read_Null               ;check switch
        ret

;--- save all input

Option25
        and     _flags2, NOT UNIQUE     ;clear flag
        call    Read_Null               ;check switch
        ret

;--- history trace recover

Option64
        or      _flags2, HTRACE          ;set flag
        call    Read_Null               ;check switch
        ret

;--- no history trace recover

Option65
        and     _flags2, NOT HTRACE      ;clear flag
        call    Read_Null               ;check switch
        ret

;--- set lock

Option44
        or      _flags4, HLOCK           ;set flag
        call    Read_Null               ;check switch
        ret

;--- clear lock

Option45
        and     _flags4, NOT HLOCK       ;clear flag
        call    Read_Null               ;check switch
        ret

;--- clear history

Option43
        mov     ax, _hisbeg
        mov     _hisend, ax             ;reset pointer
        mov     _curhis, 0              ;zero entry
        call    Read_Null               ;check switch
        ret

;--- read history from file

Option42
        call    Read_Name               ;retrieve name
        jc      op42c
        call    File_Load               ;load file
        jc      op42e
        push    cx
        push    si
        jmps    op42b
op42a   mov     di, si
        call    His_Def                 ;define line
op42b   call    File_Read               ;read a line
        jc      op42d
        jnz     op42a
        mov     di, OFFSET messS        ;exit message
        pop     si
        pop     cx
op42c   ret
op42d   pop     ax
        pop     ax
op42e   mov     bp, OFFSET messT        ;message
        ret

;--- save command history to a file

Option41
        push    si
        call    File_Reset              ;reset pointers
        mov     di, _hisbeg             ;start of history
        jmps    op41b
op41a   mov     ax, di                  ;source address
        call    File_Write              ;write line
        jc      op41c
        mov     ax, OFFSET crlf
        call    File_Write              ;write end of line
        call    Str_Next                ;next entry
op41b   cmp     di, _hisend
        jne     op41a
        call    Read_Name               ;retrieve name
        jc      op41d
        call    File_Save               ;save to file
        jnc     op41d
op41c   mov     bp, OFFSET messP
op41d   mov     di, OFFSET messO        ;exit message
        pop     si
        ret

;--- define macro (alternate syntax)

Option36
        mov     mactyp, DEFANY          ;macro type
        call    Read_Def                ;read macro definition
        jc      op36a
        push    ax
        push    bx
        call    Mac_Undef               ;undefine macro
        pop     bx
        pop     ax
        cmp     BYTE [bx], 0            ;check if just undefine
        je      op36a
        call    Mac_Def                 ;define macro
        jnc     op36a
        mov     bp, OFFSET mess3        ;message
op36a   ret

;--- macro size

Option04
        mov     ax, 10                  ;base 10
        call    Read_Word               ;read number
        mov     heap_m, ax              ;save it
        ret

;--- macro nest

Option29
        mov     ax, 10                  ;base 10
        call    Read_Byte               ;read number
        mov     BYTE _macnes, al        ;save it
        ret

;--- ignore extra arguments (autosink on)

Option95
        or      _flags4, ASINK          ;set flag
        call    Read_Null               ;check switch
        ret

;--- append extra arguments (autosink off)

Option96
        and     _flags4, NOT ASINK      ;clear flag
        call    Read_Null               ;check switch
        ret

;--- define macro

Option26
        mov     mactyp, DEFMAC          ;macro type
        call    Read_DStr               ;read double string
        jc      op26a
        call    Mac_Def                 ;define macro
        jnc     op26a
        mov     bp, OFFSET mess3        ;message
op26a   ret

;--- exchange macro (undefine then define)

Option33
        mov     mactyp, DEFMAC          ;macro type
        call    Read_DStr               ;read double string
        jc      op33a
        call    Mac_Exchg               ;exchange
        jnc     op33a
        mov     bp, OFFSET mess3        ;message
op33a   ret

;--- undefine a macro (single definition)

Option28
        mov     mactyp, DEFMAC          ;macro type
        call    Read_Str                ;read string
        jc      op28a
        call    Mac_Undef               ;define macro
        jnc     op28a
        mov     bp, OFFSET messA        ;message
op28a   ret

;--- forget a macro (all definitions)

Option31
        mov     mactyp, DEFMAC          ;macro type
        call    Read_Str                ;read string
        jc      op31c
        call    Mac_Forget              ;forget
        jnc     op31c
        mov     bp, OFFSET messY        ;message
op31c   ret

;--- clear all macros

Option30
        mov     mactyp, DEFMAC          ;macro type
        call    Mac_Purge               ;clear
        call    Read_Null               ;check switch
        ret

;--- read macros

Option37
        mov     mactyp, DEFMAC          ;macro type
        call    Read_Name               ;retrieve name
        jc      op37a
        call    Mac_Read                ;read macros
        jnc     op37a
        dec     ax
        jz      op37b
        dec     ax
        jz      op37c
        dec     ax
        jz      op37d
        stc
op37a   mov     di, OFFSET messK        ;exit message
        ret
op37b   mov     bp, OFFSET messJ        ;message
        stc
        ret
op37c   mov     bp, OFFSET messU        ;message
        stc
        ret
op37d   mov     bp, OFFSET mess8        ;message
        stc
        ret

;--- write macros

Option38
        mov     mactyp, DEFMAC          ;macro type
        call    Read_Name               ;retrieve name
        jc      op38b
        call    Mac_Write               ;write macros
        jnc     op38a
        mov     bp, OFFSET messI
op38a   mov     di, OFFSET messL        ;exit message
op38b   ret

;--- list macros

Option35
        call    Read_Null               ;check switch
        jc      op35b
        sub     di, di                  ;no exit message
        mov     mactyp, DEFMAC          ;macro type
        call    Mac_List                ;list
        jnc     op35a
        mov     di, OFFSET messH        ;no macros
op35a   clc
op35b   ret

;--- key assignment size

Option68
        mov     ax, 10                  ;base 10
        call    Read_Word               ;read number
        jc      op68b
        cmp     ax, 2                   ;check if too small
        jae     op68a
        mov     ax, 2
op68a   mov     _keysiz, ax             ;save it
        clc
op68b   ret

;--- define key assignment

Option76
        mov     mactyp, DEFKEY          ;macro type
        call    Read_DStr               ;read double string
        jc      op76a
        call    Mac_Def                 ;define macro
        jnc     op76a
        mov     bp, OFFSET mess3        ;message
op76a   ret

;--- exchange key assignment

Option77
        mov     mactyp, DEFKEY          ;macro type
        call    Read_DStr               ;read double string
        jc      op77a
        call    Mac_Exchg               ;exchange
        jnc     op77a
        mov     bp, OFFSET mess3        ;message
op77a   ret

;--- undefine a key assignment (single definition)

Option78
        mov     mactyp, DEFKEY          ;macro type
        call    Read_Str                ;read string
        jc      op78a
        call    Mac_Undef               ;define macro
        jnc     op78a
        mov     bp, OFFSET messAA       ;message
op78a   ret

;--- forget a key assignment (all definitions)

Option79
        mov     mactyp, DEFKEY          ;macro type
        call    Read_Str                ;read string
        jc      op79c
        call    Mac_Forget              ;forget
        jnc     op79c
        mov     bp, OFFSET messX        ;message
op79c   ret

;--- clear keystroke macros

Option69
        mov     mactyp, DEFKEY          ;macro type
        call    Mac_Purge               ;clear
        call    Read_Null               ;check switch
        ret

;--- read key assignments

Option75
        mov     mactyp, DEFKEY          ;macro type
        call    Read_Name               ;retrieve name
        jc      op75a
        call    Mac_Read                ;read macros
        jnc     op75a
        dec     ax
        jz      op75b
        dec     ax
        jz      op75c
        dec     ax
        jz      op75d
        stc
op75a   mov     di, OFFSET messEE       ;exit message
        ret
op75b   mov     bp, OFFSET messFF       ;message
        stc
        ret
op75c   mov     bp, OFFSET messB        ;message
        stc
        ret
op75d   mov     bp, OFFSET mess8        ;message
        stc
        ret

;--- write key assignments

Option72
        mov     mactyp, DEFKEY          ;macro type
        call    Read_Name               ;retrieve name
        jc      op72b
        call    Mac_Write
        jnc     op72a
        mov     bp, OFFSET messCC
op72a   mov     di, OFFSET messBB       ;exit message
op72b   ret

;--- list key assignments

Option71
        call    Read_Null               ;check switch
        jc      op71b
        sub     di, di                  ;no exit message
        mov     mactyp, DEFKEY          ;macro type
        call    Mac_List                ;list macros
        jnc     op71a
        mov     di, OFFSET messDD
op71a   clc
op71b   ret

;--- key input type

Option92
        mov     ax, 10                  ;base 10
        call    Read_Byte               ;read number
        jc      op92b
        cmp     al, 7                   ;check if type 7
        je      op92a
        cmp     al, 8                   ;check if type 8
        jne     op92b
op92a   mov     _keyfun, al             ;save function
        ret
op92b   stc
        ret

;--- name data size

Option70
        mov     ax, 10                  ;base 10
        call    Read_Word               ;read number
        jc      op70b
        cmp     ax, XHEAP               ;minimum bytes for matching
        jae     op70a
        sub     ax, ax                  ;zero if too small
op70a   mov     _namsiz, ax             ;save it
        clc
op70b   ret

;--- name match file types

Option46
        mov     ax, 16                  ;base 16
        call    Read_Byte               ;read number
        mov     ah, 0                   ;high byte always zero
        mov     _namtyp, ax             ;save it
        ret

;--- insert file names in lower case

Option53
        mov     _namcas, OFFSET To_Lower
        call    Read_Null               ;check switch
        ret

;--- insert file names in upper case

Option54
        mov     _namcas, OFFSET To_Upper
        call    Read_Null               ;check switch
        ret

;--- bare directory names

Option82
        and     _flags4, NOT SLASHED    ;clear flag
        call    Read_Null               ;check switch
        ret

;--- marked directory names

Option83
        or      _flags4, SLASHED        ;set flag
        call    Read_Null               ;check switch
        ret

;--- variable nest

Option84
        mov     ax, 10                  ;base 10
        call    Read_Byte               ;read number
        mov     _varmax, al             ;save it
        ret

;--- variable length

Option85
        mov     ax, 10                  ;base 10
        call    Read_Word               ;read number
        mov     _varlen, ax             ;save it
        ret

;--- set selection window columns

Option59
        mov     ax, 10                  ;base 10
        call    Read_Byte               ;read number
        mov     wincols, al             ;save it
        ret

;--- set selection window rows

Option60
        mov     ax, 10                  ;base 10
        call    Read_Byte               ;read number
        mov     winrows, al             ;save it
        ret

;--- set selection window column

Option93
        mov     ax, 10                  ;base 10
        call    Read_Byte               ;read number
        mov     wincol, al              ;save it
        ret

;--- set selection window row

Option94
        mov     ax, 10                  ;base 10
        call    Read_Byte               ;read number
        mov     winrow, al              ;save it
        ret

;--- select history entry

Option58
        call    Read_Null               ;check switch
        jc      op58a
        sub     di, di                  ;no exit message
        push    cx
        call    Select                  ;select command
        pop     cx
op58a   ret
        ENDP

;************************************************************************
; Installation Routines

;========================================================================
; Display banner.

Show_Banner PROC NEAR
        test    flags5, QUIET           ;check if skip
        jnz     shoban1
        test    sflags, BAN OR HID      ;check if skip
        jnz     shoban1
        mov     ax, OFFSET banner       ;banner
        call    MesPutL                 ;display
shoban1 ret
        ENDP

;========================================================================
; Initialize cursor scan patterns.

Default_Cur PROC NEAR
        push    cx
        push    di
        push    si
        mov     ah, 0FH                 ;get active page
        int     10H                     ;execute
        mov     ah, 3                   ;get cursor
        int     10H                     ;execute
        mov     _cdef, cx               ;save
        pop     si
        pop     di
        pop     cx
        ret
        ENDP

;========================================================================
; Remove the last command if bare TODDY.

Rem_Toddy PROC  NEAR
        push    di
        push    si
        mov     di, _hisend             ;load end
        cmp     di, _hisbeg             ;load beginning
        je      remtod4

        call    Str_Prev                ;get first entry
        mov     si, di
        mov     bx, si
        jmps    remtod2

remtod1 inc     si                      ;increment pointer
remtod2 cmp     BYTE [si], ' '          ;check if preceding space
        je      remtod1                 ;loop if so

        mov     di, OFFSET sname        ;command string

remtod3 lodsb                           ;load byte
        call    To_Upper                ;convert to uppercase
        mov     ah, [di]                ;load byte
        inc     di                      ;increment pointer
        cmp     al, ah                  ;check if same
        jne     remtod4
        or      al, al                  ;check if end of string
        jnz     remtod3                 ;loop if not
        mov     _hisend, bx             ;save new end of history
remtod4 pop     si
        pop     di
        ret
        ENDP

;========================================================================
; Determine file names.

Set_Names PROC  NEAR
        push    di
        push    si

;--- save names

        call    EnvPro                  ;get program name
        jnc     setnam1
        mov     dx, ds
        mov     ax, OFFSET tname        ;default name

setnam1 push    ds
        mov     di, pname               ;program name
        mov     si, ax                  ;environment or default name
        mov     ds, dx                  ;
        call    Copy_Str                ;copy string
        pop     ds
        sub     al, al                  ;NUL
        stosb                           ;store store it

        mov     di, cname               ;configure name
        mov     si, pname               ;source name
        call    Copy_Str                ;copy string

;--- replace extension in configuration name

        mov     bx, di                  ;save location
setnam2 cmp     di, cname               ;check at edge (no extension)
        je      setnam3
        dec     di                      ;reverse
        mov     al, [di]                ;load byte
        cmp     al, ':'                 ;check if colon (no extension)
        je      setnam3
        cmp     al, '\'                 ;check if backslash (no extension)
        je      setnam3
        cmp     al, '.'                 ;check if period (extension)
        jne     setnam2
        mov     bx, di                  ;copy over extension

setnam3 mov     di, bx                  ;target
        mov     si, OFFSET ename        ;extension
        call    Copy_Str                ;copy
        sub     al, al                  ;NUL
        stosb                           ;store store it
        pop     si
        pop     di
        ret
        ENDP

;========================================================================
; Write the settings to the executable.

Write_Opts PROC NEAR
        push    di

;--- calculate macro tail size

        mov     ax, _macend
        sub     ax, _macbeg             ;
        mov     mtail, ax               ;save tail size

;--- open file

        mov     ax, pname               ;file name
        mov     cl, OPEN_WRITE          ;open type
        call    FilOpn                  ;open file
        jc      wrtopt2
        mov     di, ax

;--- write configuration data

        mov     ax, OFFSET WRITE_BEG - 100H ;data location
        sub     dx, dx                  ;
        mov     cl, SEEK_FRONT          ;move from front
        mov     bx, di                  ;handle
        call    FilSee                  ;seek
        jc      wrtopt2

        mov     ax, WORD _flags2        ;save flags2 and flags4
        push    ax
        and     WORD _flags2, NOT (256 * DOSV32 OR FERROR OR REDISP OR FBEGUN OR HRESET)
        mov     ax, OFFSET WRITE_BEG    ;write address
        mov     dx, ds                  ;
        mov     cx, WRITE_SIZE          ;bytes to write
        mov     bx, di                  ;handle
        call    FilWri                  ;write
        pop     ax
        mov     WORD _flags2, ax        ;reset flags
        jc      wrtopt2

;--- write macros

        mov     ax, OFFSET TRAN_END - 100H ;data location
        sub     dx, dx                  ;
        mov     cl, SEEK_FRONT          ;move from front
        mov     bx, di                  ;handle
        call    FilSee                  ;seek
        jc      wrtopt2

        mov     ax, _macbeg             ;start of macros
        mov     dx, ds                  ;
        mov     cx, mtail               ;bytes to write
        mov     bx, di                  ;handle
        call    FilWri                  ;write
        jc      wrtopt2

;--- write zero bytes to truncate file

        cmp     mtail, 0                ;check if already done
        je      wrtopt1

        sub     cx, cx                  ;zero bytes
        mov     bx, di                  ;handle
        call    FilWri                  ;write
        jc      wrtopt2

;--- close file

wrtopt1 mov     bx, di                  ;handle
        call    FilClo                  ;close file
        jc      wrtopt2

        pop     di
        mov     di, OFFSET messC        ;success message
        ret

;--- error

wrtopt2 pop     di
        mov     bp, OFFSET messD        ;error message
        ret
        ENDP

;========================================================================
; Shrink the heap to the defined size.
;
; In: AX= new size.

Heap_Shrink PROC NEAR
        mov     si, _macbeg             ;start of macros
        mov     di, si
        sub     di, ax
        mov     cx, ax
        call    Heap_Adjust             ;adjust heap size
        jc      heashr1
        mov     _macbeg, di             ;new beginning
        mov     cx, _macend             ;end of macros
        sub     cx, si                  ;calculate bytes to move
        cld
        rep
        movsb
        mov     _macend, di             ;new end
heashr1 mov     bp, OFFSET messQ        ;error message if error
        ret
        ENDP

;========================================================================
; Adjust heap space.
;
; In: CX= free bytes needed.
;
; Out: CY= set if unavailable.

Heap_Adjust PROC NEAR
        push    di
        push    si
        mov     ax, _macbeg             ;start of macros
        sub     ax, _hisbeg             ;get maximum free bytes
        cmp     ax, cx                  ;check if enough
        jb      heaadj5
        jmps    heaadj4

heaadj1 push    cx
        mov     di, _hisbeg             ;start of history (start of heap)
        mov     si, di
        call    Str_Next                ;next string
        xchg    si, di                  ;swap, copy over first string

        mov     ax, _curhis             ;load last entry
        sub     ax, si                  ;get offset from start
        jb      heaadj2
        add     ax, di                  ;new address
        jmps    heaadj3
heaadj2 sub     ax, ax                  ;zero last entry
heaadj3 mov     _curhis, ax             ;save new address

        mov     cx, _hisend             ;end of history
        sub     cx, si                  ;get bytes to shift
        cld
        rep
        movsb                           ;copy bytes
        mov     _hisend, di             ;save new end
        pop     cx

heaadj4 mov     ax, _macbeg             ;start of macros
        sub     ax, _hisend             ;get free bytes
        cmp     ax, cx                  ;check if enough
        jb      heaadj1

heaadj5 pop     si
        pop     di
        ret
        ENDP

;************************************************************************
; Configuration Routines

;========================================================================
; Configure Toddy.

Configure PROC  NEAR
        call    Set_Names               ;set default names
        call    File_Alloc              ;allocate read/write buffer
        jc      config5

;--- load default configuration

        cmp     rseg, 0                 ;check resident
        jne     config1

        or      iflags, DEFCFG          ;set load flag
        mov     ax, cname               ;file name
        call    Load_Config             ;load default configuration
        jc      config6
        and     iflags, NOT DEFCFG      ;clear load flag

;--- itemize command line switches

config1 mov     sflags, 0               ;zero switch flags
        mov     si, 80H                 ;start of arguments
        cld
        lodsb                           ;load length
        mov     cl, al
        sub     ch, ch                  ;byte count in CX
        call    Check_Help              ;check for single ?
        call    Item_Switches           ;itemize switches
        jc      config6
        sub     ax, ax                  ;level 0

;--- process switches for each level

config2 push    ax
        call    Read_Switches           ;read switches
        pop     ax
        jc      config6
        test    sflags, QUI             ;check if quit
        jnz     config3
        inc     ax                      ;next level
        cmp     ax, 7                   ;check finished
        jbe     config2

;--- select window

config3 cmp     rseg, 0                 ;check if resident
        je      config4
        test    iflags, PREAD           ;check if no parameters
        jnz     config4
        sub     di, di                  ;no exit message
        or      sflags, HID             ;suppress banner
        call    Rem_Toddy               ;remove select command
        call    Select                  ;select input
        jc      config6

;--- finished

config4 call    File_Free               ;release file memory
        clc
        ret

;--- error

config5 mov     bp, OFFSET messR        ;memory error
config6 call    File_Free               ;release buffer memory
        stc
        ret
        ENDP

;========================================================================
; Check for a single ? on the command line.

Check_Help PROC NEAR
        push    cx
        push    si
        call    Skip_Delim              ;skip initial delimiters
        jc      chkhel1
        lodsb                           ;load byte
        dec     cx                      ;adjust count
        cmp     al, '?'                 ;check if question mark
        jne     chkhel1
        call    Skip_Delim              ;skip remaining command line
        jc      chkhel2                 ;display help if no more characters
chkhel1 pop     si
        pop     cx
        ret

;--- display help

chkhel2 or      iflags, PREAD           ;read a parameter
        call    Option27                ;display help
        or      sflags, QUI OR HID      ;set switch flags manually
        pop     ax                      ;bump saved SI and CX
        pop     ax
        ret
        ENDP

;========================================================================
; Load configuration from a file.
;
; In: AX= file name.
;
; Out: CY= set if error.

Load_Config PROC NEAR
        call    File_Load               ;load configuration file
        jc      lodcfg4
        push    argtab
        push    argptr
        push    arglev
        or      iflags, LODCFG          ;set flag
        mov     ax, argtab2             ;switch to argument table 2
        mov     argtab, ax              ;
        sub     ax, ax                  ;zero level
        jmps    lodcfg2

;--- process every line for every level

lodcfg1 call    Skip_Delim              ;skip delimiters
        jc      lodcfg2
        cmp     BYTE [si], COMMENT      ;check if comment line
        je      lodcfg2
        push    ax
        call    Item_Switches           ;itemize switches
        pop     ax
        jc      lodcfg3
        push    ax
        call    Read_Switches           ;process switches
        pop     ax
        jc      lodcfg3
lodcfg2 push    ax
        call    File_Read               ;read a line
        pop     ax
        jc      lodcfg3
        jne     lodcfg1

        push    ax
        call    File_Reset              ;reset file
        pop     ax
        inc     ax                      ;next level
        cmp     ax, 7                   ;check if done
        jbe     lodcfg2

;--- finished

        and     iflags, NOT LODCFG      ;clear flag
lodcfg3 pop     arglev
        pop     argptr
        pop     argtab
        ret

;--- couldn't load file

lodcfg4 test    iflags, DEFCFG          ;check if loading default config
        jnz     lodcfg5                 ;jump so (no error)
        stc
lodcfg5 ret
        ENDP

;========================================================================
; Save configuration to a file.  Note: this routine will not detect a
; failure on File_Write, only on File_Save.
;
; In: AX= file name.
;
; Out: CY= set if error.

Save_Config PROC NEAR
        push    ax
        call    File_Reset              ;reset pointers

        mov     ax, OFFSET title
        call    File_Write

        SAVFLG  'E ', 'D ', ENABLE, _flags
        SAVFLG  'Q ', 'V ', QUIET, flags5
        SAVDEC2 'B ', _inpsiz
        SAVDEC1 'BS', _sizchk
        SAVFLG  'US', 'UI', CASE, _flags
        SAVFLG  'WP', 'WU', PRIVATE, _flags4
        SAVFLG  'IR', 'IP', IRESET, _flags
        SAVFLG  'II', 'IO', IVALUE, _flags
        SAVDEC1 'CB', field1
        SAVDEC1 'CE', field2
        SAVDEC1 'CL', _liter
        SAVDEC1 'CP', _param
        SAVDEC1 'CC', _chain
        SAVDEC1 'CQ', _remove
        SAVDEC1 'CX', _quote1
        SAVDEC1 'CY', _quote2
        SAVDEC1 'CM', _allarg
        SAVDEC1 'CV', _varchr1
        SAVDEC1 'CW', _varchr2
        SAVHEX1 'AT', _atext
        SAVHEX1 'AC', _acurs
        SAVHEX1 'AE', _aend
        SAVHEX1 'AB', winatr1
        SAVHEX1 'AH', winatr2
        SAVHEX1 'AX', winatr3
        SAVHEX2 'TI', _cins
        SAVHEX2 'TO', _cover
        SAVHEX2 'TX', _cexit
        SAVFLG  'TC', 'TS', CURSOR, _flags
        SAVDEC2 'HS', heap_h
        SAVDEC1 'HM', _minlen
        SAVFLG  'HP', 'HI', PURHIS, _flags2
        SAVFLG  'HD', 'HK', DELHIS, _flags2
        SAVFLG  'HU', 'HA', UNIQUE, _flags2
        SAVFLG  'HT', 'HN', HTRACE, _flags2
        SAVFLG  'HL', 'HF', HLOCK, _flags4
        SAVDEC2 'MS', heap_m
        SAVDEC2 'MN', _macnes
        SAVFLG  'MI', 'MA', ASINK, _flags4
        SAVDEC2 'KS', _keysiz
        SAVDEC1 'KT', _keyfun
        SAVDEC2 'NS', _namsiz
        SAVHEX2 'NT', _namtyp

        mov     ax, 'NL'
        cmp     _namcas, OFFSET To_Lower
        je      savcfg1
        mov     ax, 'NU'
savcfg1 call    _wrtswi
        call    _wrteol

        SAVFLG  'NB', 'NC', SLASHED, _flags4
        SAVDEC1 'VN', _varmax
        SAVDEC2 'VL', _varlen
        SAVDEC1 'SC', wincols
        SAVDEC1 'SR', winrows
        SAVDEC1 'SX', wincol
        SAVDEC1 'SY', winrow

        mov     ax, 'KC'
        call    _wrtswi
        call    _wrteol
        mov     mactyp, DEFKEY
        mov     ax, OFFSET _wrtkey
        call    Mac_Loop

        mov     ax, 'MC'
        call    _wrtswi
        call    _wrteol
        mov     mactyp, DEFMAC
        mov     ax, OFFSET _wrtmac
        call    Mac_Loop

        pop     ax
        call    File_Save
        ret
        ENDP

;------------------------------------------------------------------------
; Save one of two switches.

;--- macro to save a switch

SAVFLG  MACRO   s1, s2, f, v
        mov     ax, s1                  ;first switch
        mov     bx, s2                  ;second switch
        test    v, f                    ;set ZF
        call    _savflg                 ;save flag
        ENDM

;--- routine to save a switch

_savflg PROC    NEAR
        jnz     savflg1                 ;check if set
        mov     ax, bx                  ;use other flag
savflg1 call    _wrtswi                 ;write the switch
        call    _wrteol                 ;end of line
        ret
        ENDP

;------------------------------------------------------------------------
; Save a switch and a decimal number.

;--- macro to save a switch and an 8-bit number

SAVDEC1 MACRO   s, v
        mov     ax, s                   ;switch
        mov     bl, v                   ;number
        sub     bh, bh                  ;
        call    _savdec                 ;save
        ENDM

;--- macro to save a switch and a 16-bit number

SAVDEC2 MACRO   s, v
        mov     ax, s                   ;switch
        mov     bx, v                   ;number
        call    _savdec                 ;save
        ENDM

;--- routine to save a switch and a decimal number

_savdec PROC    NEAR
        push    bx
        call    _wrtswi                 ;write switch
        pop     ax
        call    _wrtdec                 ;convert to string
        call    _wrteol                 ;write end of line
        ret
        ENDP

;------------------------------------------------------------------------
; Save a switch and an 8-bit hex number.

;--- macro to save an 8-bit hex number

SAVHEX1 MACRO   s, v
        mov     ax, s                   ;switch
        mov     bl, v                   ;value
        call    _savhex1                ;save
        ENDM

;--- routine to save an 8-bit hex number

_savhex1 PROC   NEAR
        push    bx
        call    _wrtswi                 ;write switch
        pop     ax
        sub     ah, ah                  ;zero high byte
        mov     cx, 2                   ;digits
        call    _wrthex                 ;write number
        call    _wrteol                 ;write end of line
        ret
        ENDP

;------------------------------------------------------------------------
; Save a switch and a 16-bit hex number.

;--- macro to save a 16-bit hex number

SAVHEX2 MACRO   s, v
        mov     ax, s                   ;switch
        mov     bx, v                   ;value
        call    _savhex2                ;save
        ENDM

;--- routine to save a 16-bit hex number

_savhex2 PROC   NEAR
        push    bx
        call    _wrtswi                 ;write switch
        pop     ax
        mov     cx, 4                   ;digits
        call    _wrthex                 ;write number
        call    _wrteol                 ;write end of line
        ret
        ENDP

;------------------------------------------------------------------------
; Write routines.

;--- write a switch

_wrtswi PROC    NEAR
        cld
        push    di
        mov     di, OFFSET numbuf       ;use number buffer for switch
        push    di
        mov     BYTE [di], SWITCH       ;store switch character
        inc     di
        stosb                           ;store first byte of switch
        mov     al, ah                  ;high byte
        cmp     al, ' '                 ;check if space
        je      wrtswi1                 ;skip if so
        stosb                           ;store second character
wrtswi1 sub     al, al                  ;NUL
        stosb                           ;save it
        pop     ax
        call    File_Write              ;write
        pop     di
        ret
        ENDP

;--- write a decimal number

_wrtdec PROC    NEAR
        mov     bx, OFFSET numbuf       ;buffer
        push    bx
        mov     BYTE [bx], ':'          ;save a colon
        inc     bx
        mov     cx, 10                  ;base 10
        sub     dx, dx                  ;zero high word
        call    Num2Str                 ;convert to string
        pop     ax
        call    File_Write              ;write
        ret
        ENDP

;--- write a hex number

_wrthex PROC   NEAR
        push    cx
        mov     bx, OFFSET numbuf       ;buffer
        mov     BYTE [bx], ':'          ;save a colon
        inc     bx                      ;go past colon
        mov     cx, 16                  ;base 16
        sub     dx, dx                  ;zero high word
        push    bx
        call    Num2Str                 ;convert to string
        pop     ax
        pop     cx
        mov     dl, '0'                 ;pad with zeros
        push    ax
        call    StrJusR                 ;justify
        pop     ax
        dec     ax                      ;include colon
        call    File_Write              ;write
        ret
        ENDP

;--- write a macro

_wrtmac PROC    NEAR
        push    bx
        push    ax
        call    _delim                  ;set delimiter
        mov     ax, 'MD'                ;switch
wrimac1 call    _wrtswi                 ;write
        mov     ax, OFFSET mquote
        call    File_Write              ;write first quote
        pop     ax
        call    File_Write              ;write macro name
        mov     ax, OFFSET mquote
        call    File_Write              ;write second quote
        pop     ax
        call    File_Write              ;write macro body
        mov     ax, OFFSET mquote
        call    File_Write              ;write third quote
        call    _wrteol                 ;write end of line
        ret
        ENDP

;--- write a key

_wrtkey PROC    NEAR
        push    bx
        push    ax
        call    _delim                  ;set delimiter
        mov     ax, 'KD'                ;switch
        jmps    wrimac1                 ;shortcut to macro write
        ENDP

;--- write an end of line marker

_wrteol PROC    NEAR
        mov     ax, OFFSET crlf         ;end of line string
        call    File_Write              ;write it
        ret
        ENDP

;------------------------------------------------------------------------
; Find a valid character for delimiting a macro.

;--- store a valid delimiter for macro in AX/BX

_delim  PROC   NEAR
        mov     dl, 32                  ;starting character - 1
delim1  xchg    ax, bx
        call    _valdel                 ;find delimiter
        xchg    ax, bx
        call    _valchk                 ;check if okay
        jc      delim1
        mov     mquote, dl              ;save it
        ret
        ENDP

;--- find a next valid delimiter for the string in AX

_valdel PROC    NEAR
valdel1 inc     dl                      ;next character
        call    _valchk                 ;check if valid
        jc      valdel1
        ret
        ENDP

;--- check if DL is valid delimiter for string in AX

_valchk PROC    NEAR
        push    ax
        push    si
        mov     si, ax                  ;source
        jmps    valchk2
valchk1 cmp     al, dl                  ;check if match
        stc                             ;set carry if error
        je      valchk3
valchk2 cld
        lodsb                           ;load next byte
        or      al, al                  ;check if end of string
        jnz     valchk1                 ;loop back if not
        clc
valchk3 pop     si
        pop     ax
        ret
        ENDP

;************************************************************************
; Switch Routines

;========================================================================
; Advance to next switch and load command.
;
; In: SI= pointer; CX= bytes.
;
; Out: AX= switch code ('  ' if no slash); SI,CX= updated; CY= set if
;      no more switches.

Next_Switch PROC NEAR
        call    Skip_Delim              ;skip delimiters
        jc      nexswi3
        mov     ax, '  '                ;non-slash switch
        cmp     BYTE [si], SWITCH       ;check if switch character
        jne     nexswi2
        inc     si                      ;skip slash
        dec     cx
        jz      nexswi3
        cld
        lodsb                           ;load first character
        call    To_Upper                ;convert to upper case
        mov     ah, ' '                 ;default second character
        dec     cx                      ;decrement count
        jz      nexswi2

        mov     dl, al                  ;save low character
        mov     al, [si]                ;load second character
        call    To_Upper                ;convert to upper case
        cmp     al, 'A'                 ;check if letter
        jb      nexswi1
        cmp     al, 'Z'                 ;check if letter
        ja      nexswi1
        mov     ah, al                  ;use letter
        inc     si                      ;increment pointer
        dec     cx                      ;decrement bytes
nexswi1 mov     al, dl
nexswi2 mov     serror, ax              ;save for error
        clc
        ret

nexswi3 stc
        ret
        ENDP

;========================================================================
; Look up switch.
;
; In: AX= switch.
;
; Out: BX= address of switch data; CY= set if not found.

Find_Switch PROC NEAR
        mov     bx, OFFSET switches     ;table offset
finswi1 cmp     ax, [bx]                ;check if switch matches
        je      finswi2
        add     bx, 6                   ;skip entry
        cmp     BYTE [bx], 0            ;check if end of switches
        jne     finswi1
        stc
finswi2 ret
        ENDP

;========================================================================
; Find end of argument.
;
; In: SI= pointer; CX= bytes.
;
; Out: SI,CX= updated; CY= set error; ZF= set finished (if no error).

Skip_Switch PROC NEAR
        call    Next_Switch             ;next argument
        jc      skiswi2                 ;exit if none
        call    Find_Switch             ;look up command
        jc      skiswi3                 ;exit if error
        jcxz    skiswi2                 ;exit if no more bytes
        mov     al, [bx + 5]            ;load value type
        and     al, 3                   ;mask bits
        dec     al                      ;check if single string
        jz      skiswi4
        dec     al                      ;check if double string
        jz      skiswi6
        dec     al                      ;check if skip remaining
        jz      skiswi7

;--- skip nothing, a number, or a file name

        cld
skiswi1 jcxz    skiswi2
        lodsb                           ;load byte
        dec     cx                      ;decrement bytes remaining
        cmp     al, ' '                 ;check if delimiter
        jbe     skiswi2
        cmp     al, SWITCH              ;check if start of switch
        jne     skiswi1

        dec     si                      ;reread last character
        inc     cx                      ;
skiswi2 or      cx, cx                  ;set ZF
        clc                             ;no error
skiswi3 ret

;--- skip a single string

skiswi4 cld
        lodsb                           ;load quote character
        dec     cx                      ;
        call    Skip_Token              ;skip token
skiswi5 jc      skiswi3                 ;jump if error
        jmps    skiswi2

;--- skip a double string

skiswi6 cld
        lodsb                           ;load quote character
        dec     cx                      ;
        call    Skip_Token              ;skip token
        jc      skiswi3                 ;jump if error
        call    Skip_Token              ;skip token
        jmps    skiswi5

;--- skip remaining characters

skiswi7 add     si, cx                  ;advance pointer
        sub     cx, cx                  ;zero bytes
        jmps    skiswi2
        ENDP

;========================================================================
; Itemize a line of switches.
;
; In: SI= pointer; CX= bytes.
;
; Out: CY= set if error.

Item_Switches PROC NEAR
        push    di
        push    bp
        mov     di, argtab              ;start of table
        sub     bp, bp                  ;zero count
        jmps    itmswi2
itmswi1 cmp     bp, MAXARG              ;check if too many arguments
        stc                             ;set carry for error
        je      itmswi3                 ;exit if error
        inc     bp                      ;increment argument count
itmswi2 mov     [di], si                ;save address
        inc     di
        inc     di
        mov     [di], cx                ;save bytes
        inc     di
        inc     di
        call    Skip_Switch             ;next switch
        jc      itmswi3
        jnz     itmswi1
        mov     WORD [di], 0            ;store a NUL to the end of list
        mov     serror, 0               ;clear error switch
itmswi3 pop     bp
        pop     di
        ret
        ENDP

;========================================================================
; Process itemized switches.
;
; In: AX= switch level to process.

Read_Switches PROC NEAR
        mov     arglev, ax              ;save level
        mov     ax, argtab              ;start of table
        mov     argptr, ax              ;

reaswi1 push    di
        mov     di, argptr              ;load pointer
        mov     si, [di]                ;load source
        inc     di
        inc     di
        mov     cx, [di]                ;load bytes (invalid if si=0)
        inc     di
        inc     di
        mov     argptr, di              ;save pointer
        pop     di
        or      si, si                  ;check if finished
        jz      reaswi6

        call    Next_Switch             ;next argument
        jc      reaswi6
        call    Find_Switch             ;look up switch
        jc      reaswi7
        mov     al, [bx + 4]            ;load level
        and     ax, 7                   ;mask level
        cmp     ax, arglev              ;check if level matches
        jne     reaswi1

        mov     al, [bx + 5]            ;load flags
        test    al, CFG                 ;check if can't be loading config
        jz      reaswi2
        test    iflags, LODCFG          ;check if loading configuration
        jnz     reaswi8

reaswi2 test    al, RES                 ;check if must be resident
        jz      reaswi3
        cmp     rseg, 0                 ;check if resident
        je      reaswi9

reaswi3 test    al, NON                 ;check if can't be resident
        jz      reaswi4
        cmp     rseg, 0                 ;check if not resident
        jne     reaswiA

reaswi4 test    al, BAN                 ;check if show banner
        jz      reaswi5
        push    ax
        push    bx
        call    Show_Banner             ;display banner
        pop     bx
        pop     ax

reaswi5 or      sflags, al              ;set switch flags
        or      iflags, PREAD           ;set parameter flag
        call    WORD [bx + 2]           ;process switch
        jc      reaswi7
        test    sflags, QUI             ;check if quit
        jz      reaswi1                 ;loop back if not

reaswi6 mov     serror, 0               ;clear error switch
        clc
reaswi7 ret

reaswi8 mov     bp, OFFSET messZ
        stc
        ret

reaswi9 test    iflags, LODCFG          ;check if loading configuration
        jnz     reaswi5
        mov     bp, OFFSET mess6
        stc
        ret

reaswiA test    iflags, LODCFG          ;check if loading configuration
        jnz     reaswi5
        mov     bp, OFFSET mess7
        stc
        ret
        ENDP

;************************************************************************
; History Routines

;========================================================================
; Add string to the history.
;
; In: DI= string; CX= length.
;
; Out: CY= set if insufficient heap.

His_Def PROC   NEAR
        push    si
        mov     si, di
        test    _flags2, PURHIS         ;check if pure history
        jz      hisdef1
        mov     bx, OFFSET hisdef3
        call    His_Dups                ;delete duplicates
hisdef1 inc     cx                      ;include NUL
        call    Heap_Adjust             ;adjust heap space
        jc      hisdef2                 ;exit if space unavailable
        mov     di, _hisend             ;history end
        cld
        rep
        movsb                           ;copy to history
        mov     _hisend, di             ;new end
hisdef2 pop     si
hisdef3 ret
        ENDP

;========================================================================
; Purify the history.

His_Purify PROC NEAR
        push    cx
        push    di
        push    si
        cld
        mov     si, _hisend             ;start at end of history
        jmps    hispur2
hispur1 mov     di, si
        call    Str_Prev                ;goto start of line
        mov     si, di
        mov     bx, OFFSET hispur3
        call    His_Dups                ;remove duplicates
hispur2 cmp     si, _hisbeg             ;check if done
        jne     hispur1
        pop     si
        pop     di
        pop     cx
        ret
hispur3 sub     si, cx                  ;update routine
        dec     si                      ;include NUL
        ret
        ENDP

;========================================================================
; Delete duplicate entries.
;
; In: SI= string; BX= string update routine.
;
; Out: SI= updated.

His_Dups PROC   NEAR
        push    cx
        mov     di, si
        call    Str_Length              ;get length
        mov     di, _hisend             ;start at end
        jmps    hisdup2

hisdup1 call    Str_Prev                ;previous entry
        cmp     di, si                  ;check if same string
        je      hisdup2
        call    Str_Comp                ;compare macros
        jne     hisdup2
        push    cx
        push    di
        push    si
        mov     si, di                  ;save start
        call    Str_Next                ;goto end of string
        xchg    di, si                  ;set up source and destination
        mov     cx, _hisend             ;end address
        sub     cx, si                  ;get bytes to move
        rep
        movsb                           ;shift history
        mov     _hisend, di             ;new end of table
        pop     si
        pop     di
        pop     cx
        call    bx                      ;update source

hisdup2 cmp     di, _hisbeg             ;check if done
        jne     hisdup1
        pop     cx
        ret
        ENDP

;************************************************************************
; Macro Routines

;========================================================================
; Define a macro.
;
; In: AX= macro name; BX= macro text.
;
; Out: CY= set if not enough memory.

Mac_Def PROC    NEAR
        push    cx
        push    di
        push    si

        mov     di, ax
        mov     si, bx
        call    Str_Length              ;get length of symbol
        inc     cx                      ;include NUL
        push    cx
        xchg    di, si
        call    Str_Length              ;get length of text
        pop     ax
        inc     cx                      ;include NUL
        add     cx, ax                  ;add lengths together
        call    Heap_Adjust             ;release space
        jc      macdef1                 ;exit if error

        push    di
        mov     di, _macbeg             ;start of macros
        sub     di, cx                  ;new start
        mov     _macbeg, di             ;save it

        push    di
        mov     di, si
        call    Str_Length              ;get first length
        inc     cx                      ;include NUL
        pop     di

        cld
        rep
        movsb                           ;copy to table

        pop     si                      ;restore text

        push    di
        mov     di, si
        call    Str_Length              ;get length
        inc     cx                      ;include NUL
        pop     di
        cld
        rep
        movsb                           ;copy to table
        clc

macdef1 pop     si
        pop     di
        pop     cx
        ret
        ENDP

;========================================================================
; Exchange a macro.
;
; In: AX= macro name; BX= macro text.
;
; Out: CY= set if not enough memory.

Mac_Exchg PROC  NEAR
        push    ax
        push    bx
        call    Mac_Undef               ;undefine macro
        pop     bx
        pop     ax
        call    Mac_Def                 ;define macro
        ret
        ENDP

;========================================================================
; Undefine a macro.
;
; In: AX= macro name.
;
; Out: CY= set if macro not found.

Mac_Undef PROC  NEAR
        push    cx
        push    di
        push    si
        call    Mac_Locate              ;find macro
        jc      macund1
        mov     di, ax                  ;start of macro
        call    Str_Next                ;skip macro
        call    Str_Next                ;
        mov     cx, ax                  ;bytes to move
        sub     cx, _macbeg             ;
        dec     di                      ;end of macro to delete
        mov     si, ax                  ;begin with previous macro
        dec     si                      ;
        std
        rep
        movsb                           ;delete
        inc     di
        mov     _macbeg, di             ;save new beginning
        clc
macund1 pop     si
        pop     di
        pop     cx
        ret
        ENDP

;========================================================================
; Forget a macro (undefine all definitions).
;
; In: AX= macro name.
;
; Out: CY= set if no definitions.

Mac_Forget PROC NEAR
        push    di
        push    si
        mov     si, ax                  ;save name
        sub     di, di                  ;zero count
        jmps    macfor2
macfor1 inc     di                      ;increment count
macfor2 mov     ax, si
        call    Mac_Undef               ;undefine macro
        jnc     macfor1
        sub     di, 1                   ;set CY if no definitions
        pop     si
        pop     di
        ret
        ENDP

;========================================================================
; Erase all macros.

Mac_Purge PROC  NEAR
        push    di
macpur1 mov     di, _macbeg             ;start of macros
        jmps    macpur4

macpur2 call    Mac_Check               ;check if skipped
        jc      macpur3

        mov     ax, di
        call    Mac_Undef               ;delete it
        jmps    macpur1

macpur3 call    Str_Next                ;next macro
        call    Str_Next                ;

macpur4 cmp     di, _macend             ;check if done
        jne     macpur2
        pop     di
        ret
        ENDP

;========================================================================
; Display all macros.
;
; Out: CY= set if no macros defined.

Mac_List PROC   NEAR
        call    Mac_Count               ;count macros
        stc
        je      maclis1
        mov     ax, OFFSET crlf         ;blank line
        call    MesPut                  ;display
        mov     ax, OFFSET maclis2      ;display function
        call    Mac_Loop                ;macro loop
        clc
maclis1 ret

;--- list routine

maclis2 push    bx
        call    MesPut                  ;macro name
        mov     ax, OFFSET equals
        call    MesPut                  ;equals
        pop     ax
        call    MesPutL                 ;macro body
        clc
        ret
        ENDP

;========================================================================
; Read macros from a file.
;
; In: AX= file name.
;
; Out: CY= set if error; AX= error number: 1 if read error, 2 define
;      error, 3 out of memory.

Mac_Read PROC   NEAR
        push    cx
        push    di
        push    si
        mov     di, 1                   ;error number
        call    File_Load               ;load file
        jc      macrea5
        jmps    macrea4

;--- check if blank line

macrea1 push    cx
        push    si
macrea2 cld
        lodsb                           ;load byte
        cmp     al, ' '                 ;check if space
        jne     macrea3
        loop    macrea2                 ;loop while characters
macrea3 pop     si
        pop     cx
        jz      macrea4                 ;jump if blank line

;--- define macro

        inc     di
        call    Read_Def                ;read macro definition
        jc      macrea5
        inc     di
        call    Mac_Def                 ;define macro
        jc      macrea5
        dec     di
        dec     di

macrea4 call    File_Read               ;read a line
        jc      macrea5
        jnz     macrea1
macrea5 mov     ax, di
        pop     si
        pop     di
        pop     cx
        ret
        ENDP

;========================================================================
; Write macros to a file.
;
; In: AX= file name.
;
; Out: CY= set if error.

Mac_Write PROC  NEAR
        push    ax
        call    File_Reset              ;reset file
        mov     ax, OFFSET macwri2      ;write function
        call    Mac_Loop                ;loop for all macros
        pop     ax
        jc      macwri1
        call    File_Save               ;save file
macwri1 ret

;--- write routine

macwri2 push    di
        mov     di, bx
        call    File_Write              ;name
        jc      macwri3
        mov     ax, OFFSET equals
        call    File_Write              ;equals
        jc      macwri3
        mov     ax, di
        call    File_Write              ;body
        jc      macwri3
        mov     ax, OFFSET crlf
        call    File_Write              ;end of line
macwri3 pop     di
        ret
        ENDP

;========================================================================
; Determine assignment type.
;
; In: AX= string.
;
; Out: CY= set if error.

Mac_Type PROC   NEAR
        cmp     mactyp, DEFMAC          ;check if macro definition
        je      mactyp1

        push    bx
        mov     bx, ax
        mov     bl, [bx]                ;first character
        cmp     bl, field1              ;check if start of key
        pop     bx
        je      mactyp2

        cmp     mactyp, DEFKEY          ;check if key assignment
        stc
        je      mactyp1
        mov     mactyp, DEFMAC          ;save type
        clc
mactyp1 ret

mactyp2 mov     mactyp, DEFKEY          ;save type
        clc
        ret
        ENDP

;========================================================================
; Check if macro should be processed.
;
; In: DI= macro address.
;
; Out: CY= set if skip.

Mac_Check PROC  NEAR
        mov     al, [di + 1]            ;load second byte
        cmp     mactyp, DEFKEY          ;check if defining key
        je      macchk1

        cmp     al, COMMAND             ;check if command
        je      macchk2
        clc
        ret

macchk1 cmp     al, COMMAND             ;check if command
        je      macchk3
macchk2 stc
macchk3 ret
        ENDP

;========================================================================
; Count macros.
;
; Out: ZF= set if no macros.

Mac_Count PROC  NEAR
        push    di
        sub     bx, bx                  ;zero key count
        mov     di, _macbeg             ;start of table
        jmps    maccnt3

maccnt1 call    Mac_Check               ;check if should be counted
        jc      maccnt2
        inc     bx                      ;increment count
maccnt2 call    Str_Next                ;next macro
        call    Str_Next                ;
maccnt3 cmp     di, _macend             ;check if done
        jne     maccnt1

        or      bx, bx                  ;set ZF
        pop     di
        ret
        ENDP

;========================================================================
; Find a macro.
;
; In: AX= name of macro.
;
; Out: CY= set if not found.

Mac_Locate PROC NEAR
        push    cx
        push    di
        push    si
        mov     si, ax
        mov     di, ax
        call    Str_Length              ;get length of name
        mov     di, _macbeg             ;start of macros
        jmps    macloc2

macloc1 cld
        call    Str_Comp                ;compare macros
        je      macloc3
        call    Str_Next                ;skip name
        call    Str_Next                ;skip body
macloc2 cmp     di, _macend             ;check if at end
        jne     macloc1

        stc                             ;not found
macloc3 mov     ax, di                  ;return address
        pop     si
        pop     di
        pop     cx
        ret
        ENDP

;========================================================================
; Send all macros as ASCIIZ strings to a routine.  The routine should
; accept the macro name in AX and the macro body in BX, and should return
; CY set if there's an error.
;
; In: AX= routine.
;
; Out: CY= set if error in routine.

Mac_Loop PROC   NEAR
        push    cx
        push    di
        push    si

        mov     di, _macbeg             ;start of macros
        mov     BYTE [di - 1], 0        ;store a NUL for Str_Prev
        mov     di, _macend             ;end of macros
        mov     si, ax                  ;save output routine
        jmps    macloo3

macloo1 call    Str_Prev                ;go to start of macro
        call    Str_Prev                ;
        call    Mac_Check               ;check if process
        jc      macloo3

        push    di
        mov     ax, di                  ;macro name
        call    Str_Next                ;next string
        mov     bx, di                  ;macro body
        pop     di
        cmp     mactyp, DEFKEY          ;check if processing key
        jne     macloo2
        call    Key_Back                ;convert back to ASCII
macloo2 call    si                      ;process
        jc      macloo4

macloo3 cmp     di, _macbeg             ;check finished
        jne     macloo1                 ;loop back if not

macloo4 pop     si
        pop     di
        pop     cx
        ret
        ENDP

;************************************************************************
; Parsing Routines

;========================================================================
; Copy an ASCIIZ string.
;
; In: DS:SI= source; ES:DI= target.

        PROC    NEAR
copstr1 stosb                           ;store byte
Copy_Str LABEL  NEAR                    ;entry point
        cld                             ;clear direction
        lodsb                           ;load byte
        or      al, al                  ;check if NUL
        jnz     copstr1                 ;loop if not
        ret
        ENDP

;========================================================================
; Skip delimiters.
;
; In: SI= pointer; CX= bytes.
;
; Out: SI,CX= updated; CY= set if end of bytes.

Skip_Delim PROC NEAR
        jcxz    skidel2                 ;exit if none
skidel1 cmp     BYTE [si], ' '          ;check if delimiter
        ja      skidel3
        inc     si                      ;increment pointer
        loop    skidel1                 ;loop if more bytes
skidel2 stc
skidel3 ret
        ENDP

;========================================================================
; Skip a ':' or '='.
;
; In: SI= pointer; CX= bytes.
;
; Out: SI,CX= updated; ZF= set if no more bytes.

Skip_Sep PROC   NEAR
        push    ax
        jcxz    skisep2                 ;exit if no more bytes
        mov     al, [si]                ;load byte
        cmp     al, '='                 ;check if =
        je      skisep1
        cmp     al, ':'                 ;check :
        jne     skisep2
skisep1 inc     si                      ;skip byte
        dec     cx                      ;adjust count
skisep2 or      cx, cx                  ;set ZF
        pop     ax
        ret
        ENDP

;========================================================================
; Clean up a string by removing all leading and trailing spaces.
;
; In: BX= string.
;
; Out: BX= new start address; CY= set if error.

        PROC NEAR

;--- remove all leading spaces

strcle1 inc     bx                      ;next character
String_Clean LABEL NEAR
        cmp     BYTE [bx], 0            ;check if end of string
        je      strcle4
        cmp     BYTE [bx], ' '          ;check if space
        je      strcle1

        push    bx                      ;save for exit

;--- remove all trailing spaces

strcle2 inc     bx                      ;next character
        cmp     BYTE [bx], 0            ;check if end of string
        je      strcle3
        cmp     BYTE [bx], ' '          ;check if space
        jne     strcle2
        mov     BYTE [bx], 0            ;truncate at location of space

strcle3 pop     bx
        clc
        ret

strcle4 pop     bx
        stc
        ret
        ENDP

;========================================================================
; Skip a token.
;
; In: AL= end delimiter; SI= pointer; CX= bytes.
;
; Out: SI,CX= updated; CY= set if end of bytes.

Skip_Token PROC NEAR
        jcxz    skitok2                 ;exit if none
skitok1 cmp     al, [si]                ;check if delimiter
        je      skitok3
        inc     si                      ;increment pointer
        loop    skitok1                 ;loop if more bytes
skitok2 stc
        ret

skitok3 inc     si                      ;skip byte
        dec     cx                      ;reduce bytes left
        ret
        ENDP

;========================================================================
; Parse a token. A NUL is stored at the end of the token.
;
; In: AL= end delimiter; SI= pointer; CX= bytes.
;
; Out: BX= start of token; SI,CX= updated; CY= set if end of bytes.

Parse_Token PROC NEAR
        mov     bx, si                  ;save start
        call    Skip_Token              ;skip to end
        jc      partok1
        mov     BYTE [si - 1], 0        ;store NUL
partok1 ret
        ENDP

;========================================================================
; Make sure no characters after switch.
;
; Out: CY= set if error.

Read_Null PROC  NEAR
        clc                             ;clear CY
        jcxz    reanul1                 ;exit if CX zero
        cmp     BYTE [si], ' ' + 1      ;set CY if delimiter
        cmc                             ;switch carry
reanul1 ret
        ENDP

;========================================================================
; Read a 16 bit number.
;
; In: AX= base; SI= pointer; CX= bytes.
;
; Out: AX= number (zero if none); SI,CX= updated; CY= set if error.

Read_Word PROC  NEAR
        call    Skip_Sep                ;skip separator
        jz      reawrd4

        push    di
        push    si                      ;save address to check if any digits
        mov     di, ax                  ;save base
        sub     bx, bx                  ;zero total

;--- loop until end of number

reawrd1 mov     al, [si]                ;load digit
        cmp     al, ' '                 ;check if delimiter
        jbe     reawrd2
        cmp     al, SWITCH              ;check if switch
        je      reawrd2

        push    bx
        push    cx
        push    di
        call    To_Upper                ;convert to upper case
        mov     cx, di                  ;digit base
        mov     di, OFFSET digits       ;address of digits
        cld
        repne
        scasb                           ;scan for digit
        mov     ax, cx                  ;save count
        pop     di
        pop     cx
        pop     bx
        jne     reawrd3                 ;jump if error

        xchg    ax, bx
        mul     ax, di                  ;total times base
        or      dx, dx                  ;check for overflow
        jnz     reawrd3
        sub     bx, di                  ;get digit value
        neg     bx                      ;
        dec     bx                      ;
        add     bx, ax                  ;update value
        jc      reawrd3

        inc     si                      ;skip byte
        loop    reawrd1                 ;loop for each byte

reawrd2 mov     ax, bx                  ;return total
        pop     dx
        pop     di
        cmp     dx, si                  ;check if any bytes read
        je      reawrd4
        clc
        ret

reawrd3 pop     dx
        pop     di
reawrd4 stc
        ret
        ENDP

;========================================================================
; Read an 8 bit number.  Parameters just like Read_Word.
;
; Out: AL= value.

Read_Byte PROC  NEAR
        call    Read_Word               ;read a word
        jc      reabyt1                 ;exit if carry set
        sub     ah, 1                   ;decrement (set carry if zero)
        cmc                             ;carry set if not zero
reabyt1 ret
        ENDP

;========================================================================
; Read a raw quoted string.
;
; Out: BX= start of token.

Read_StrX PROC  NEAR
        jcxz    reastx1                 ;exit if no bytes left
        cld
        lodsb                           ;load quote character
        dec     cx                      ;decrement bytes
        call    Parse_Token             ;get string
        ret
reastx1 stc
        ret
        ENDP

;========================================================================
; Read a cooked quoted string.
;
; Out: AX= string; CY= set if format error.

Read_Str PROC NEAR
        call    Read_StrX               ;read a string
        jc      reastr1
        call    String_Clean            ;clean it
        mov     ax, bx
        call    Key_Str                 ;convert key string
reastr1 ret
        ENDP

;========================================================================
; Read a cooked double quoted string.  The first string is cleaned.
;
; Out: AX,BX= first and second strings; CY= set if format error.

Read_DStr PROC NEAR
        call    Read_StrX               ;read a string
        jc      readst1
        call    String_Clean            ;clean it
        push    bx
        call    Parse_Token             ;second string
        pop     ax
        call    Key_DStr                ;convert key string
readst1 ret
        ENDP

;========================================================================
; Read a macro definition.  The first string is cleaned.
;
; Out: AX,BX= first and second strings; CY= set if format error.

Read_Def PROC   NEAR
        mov     al, '='                 ;look for equals
        call    Parse_Token             ;get string
        jc      readef1
        call    String_Clean            ;clean the string
        jc      readef1
        mov     ax, bx                  ;first string
        mov     bx, si                  ;second string
        add     si, cx                  ;end pointer
        mov     BYTE [si], 0            ;store NUL to end of second string
        sub     cx, cx                  ;zero bytes remaining
        call    Key_DStr                ;convert key string
        ret
readef1 stc
        ret
        ENDP

;========================================================================
; Read a filename.
;
; Out: AX= filename; CY= set if format error (no name).

Read_Name PROC  NEAR
        mov     ax, si                  ;return start
        mov     bx, si                  ;save start
        call    Skip_Sep                ;skip separator
        jz      reanam2
reanam1 cmp     BYTE [si], ' '          ;check if delimiter
        jbe     reanam2
        inc     si                      ;increment pointer
        loop    reanam1                 ;loop if more bytes
        inc     cx                      ;adjust for DEC below
reanam2 cmp     ax, si                  ;set carry if 0 bytes
        cmc                             ;
        mov     BYTE [si], 0            ;store NUL         ;
        inc     si                      ;skip byte         ;-- CY can't be
        dec     cx                      ;reduce bytes left ;   modified
        ret
        ENDP

;========================================================================
; Convert definition string to a key string.
;
; In: AX= string.

Key_StrX PROC    NEAR
        push    bx
        push    cx
        push    di
        push    si
        push    WORD _flags
        push    ax
        and     _flags, NOT CASE        ;clear case sensitivity

        cld
        mov     si, ax
        mov     di, si
        jmps    keystx3

keystx1 cmp     al, field1              ;check if start of field
        je      keystx7
        cmp     al, COMMAND             ;check if delimiter
        je      keystxC
keystx2 stosb                           ;store byte
keystx3 lodsb                           ;load byte
        or      al, al                  ;check if end of string
        jnz     keystx1
        stosb                           ;store NUL

;--- reverse string

        pop     si                      ;first character
        push    si                      ;
        dec     di                      ;last character
        dec     di                      ;
        jmps    keystx5

keystx4 mov     al, [si]                ;load byte
        xchg    al, [di]                ;swap with end
        mov     [si], al                ;store byte
        inc     si                      ;adjust pointers
        dec     di                      ;
keystx5 cmp     si, di                  ;check if finished
        jb      keystx4

keystx6 pop     ax
        pop     WORD _flags
        pop     si
        pop     di
        pop     cx
        pop     bx
        ret

;--- find end of field

keystx7 mov     cx, si                  ;save start of field
        mov     bx, si                  ;save for number check

keystx8 lodsb                           ;load next byte
        or      al, al                  ;check if end of string
        jz      keystxC
        cmp     al, field2              ;check if end of field
        jne     keystx8

;--- find command

        push    di
        push    si
        dec     si                      ;reverse to end of field
        mov     BYTE [si], 0            ;store NUL for number check
        xchg    cx, si                  ;start of command
        sub     cx, si                  ;calculate length
        mov     di, OFFSET cmdstr       ;start of valid field strings
keystx9 mov     ah, [di]                ;load command code
        or      ah, ah                  ;check if end of table
        jz      keystxA
        inc     di                      ;skip code
        call    Str_Comp                ;check if field matches
        je      keystxB
        call    Str_Next                ;goto next entry
        jmps    keystx9

;--- command not found, check if number

keystxA pop     si
        pop     di
        mov     ax, bx                  ;string
        mov     cx, 10                  ;base 10
        call    Str2Num                 ;convert to number
        or      ax, ax                  ;check if zero
        jz      keystxC
        cmp     ax, 255                 ;check if too big
        ja      keystxC
        or      dx, dx                  ;check if too big
        jnz     keystxC
        jmps    keystx2

;--- store command

keystxB pop     si
        pop     di
        mov     al, COMMAND             ;command delimiter
        stosw                           ;store
        jmps    keystx3

;--- error

keystxC stc
        jmps    keystx6
        ENDP

;========================================================================
; Convert to a key string if necessary.
;
; In: AX= string.

Key_Str PROC    NEAR
        call    Mac_Type                ;determine type
        jc      keystr1
        cmp     mactyp, DEFMAC          ;check if macro definition
        je      keystr1
        call    Key_StrX
keystr1 ret
        ENDP

;========================================================================
; Convert to a double key string if necessary.
;
; In: AX,BX= strings.

Key_DStr PROC    NEAR
        call    Mac_Type                ;determine type
        jc      keydst1
        cmp     mactyp, DEFMAC          ;check if macro definition
        je      keydst1
        call    Key_StrX                ;convert first string
        jc      keydst1
        xchg    ax, bx
        call    Key_StrX                ;convert second string
        jc      keydst1
        xchg    ax, bx
keydst1 ret
        ENDP

;========================================================================
; Convert a key sequence to a string.
;
; In: DI= macro; AX= target.
;
; Out: AX= end of string

Key_BackX PROC  NEAR
        push    cx
        push    di
        push    si
        cld
        call    Str_Next                ;end of string
        dec     di                      ;point to end
        mov     si, di
        mov     di, ax                  ;target
        jmps    keybac2
keybac1 cmp     al, COMMAND             ;check if command
        je      keybac3
        cmp     al, field1              ;check if field character
        je      keybac9
        cmp     al, 32                  ;check if control character
        jb      keybac9
        cmp     al, 127                 ;check if extended character
        jae     keybac9
        stosb                           ;store byte
keybac2 dec     si                      ;previous byte
        mov     al, [si]                ;load source
        or      al, al                  ;check if finished
        jnz     keybac1
        stosb                           ;store NUL
        mov     ax, di                  ;return end
        pop     si
        pop     di
        pop     cx
        ret

;--- save a command string

keybac3 mov     al, field1              ;start of field
        stosb                           ;save it
        dec     si                      ;previous byte
        mov     al, [si]                ;load code
        or      al, al                  ;check if end of line
        jz      keybac8

        push    si
        push    di
        mov     di, OFFSET cmdstr       ;start of table
        jmps    keybac5
keybac4 mov     ah, [di]                ;load next code
        inc     di                      ;increment pointer
        cmp     al, ah                  ;check if match
        je      keybac7
        call    Str_Next                ;next entry
keybac5 cmp     BYTE [di], 0            ;check if end of table
        jnz     keybac4
keybac6 mov     di, OFFSET cmderr + 1   ;error string
keybac7 mov     si, di
        call    Str_Next                ;end of string
        mov     cx, di                  ;calculate length
        sub     cx, si                  ;
        dec     cx                      ;
        pop     di
        rep
        movsb                           ;copy to buffer
        pop     si
        mov     al, field2              ;end of field
        stosb                           ;save it
        jmps    keybac2

keybac8 inc     si                      ;reread end of line
        push    si
        push    di
        jmps    keybac6

;--- save a number

keybac9 push    ax
        mov     al, field1              ;start of field
        stosb                           ;save it
        pop     ax
        sub     ah, ah                  ;zero rest of number
        sub     dx, dx                  ;
        mov     bx, di                  ;place to save it
        mov     cx, 10                  ;base 10
        call    Num2Str                 ;convert to string
        add     di, ax                  ;advance pointer
        mov     al, field2              ;end of field
        stosb                           ;save it
        jmps    keybac2
        ENDP

;========================================================================
; Convert a key macro to a string.
;
; In: DI= start of macro.
;
; Out: AX= name; BX= body.

Key_Back PROC   NEAR
        push    di
        mov     ax, keybuf              ;key buffer
        push    ax
        call    Key_BackX               ;convert to string
        call    Str_Next                ;body of macro
        push    ax
        call    Key_BackX               ;convert to string
        pop     bx
        pop     ax
        pop     di
        ret
        ENDP

;************************************************************************
; Input/Output Routines

;========================================================================
; Allocate read/write buffer.
;
; Out: CY= set if error.

File_Alloc PROC  NEAR
        mov     ah, 48H                 ;allocate function
        mov     bx, 1000H               ;paragraphs to allocate
        int     21H                     ;execute
        jc      filall1                 ;check if error
        mov     filseg, ax              ;save segment
filall1 ret
        ENDP

;========================================================================
; Free read/write buffer.

File_Free PROC  NEAR
        push    es
        mov     ax, filseg              ;load segment
        or      ax, ax                  ;check if defined
        jz      filfre1                 ;exit if not
        mov     es, ax
        mov     ah, 49H                 ;release function
        int     21H                     ;execute
filfre1 pop     es
        ret
        ENDP

;========================================================================
; Reset read pointer.

File_Reset PROC  NEAR
        mov     filptr, 0               ;zero pointer
        mov     ax, fillen              ;reset remaining bytes
        mov     filrem, ax              ;
        ret
        ENDP

;========================================================================
; Read a line.
;
; Out: SI= start of line; CX= bytes in line; ZF= set if end of file (if
;      no error); CY= set if error.

File_Read PROC  NEAR
        push    di
        push    ds
        cld
        mov     di, linbuf              ;line buffer
        sub     dx, dx                  ;zero bytes read
        mov     si, filptr              ;load pointer
        mov     cx, filrem              ;load bytes
        mov     ds, filseg              ;load buffer segment
        jcxz    filrea4                 ;exit if no more bytes

filrea1 lodsb                           ;load byte
        dec     cx                      ;adjust count
        cmp     al, 10                  ;check if end of line
        je      filrea5
        cmp     al, 26                  ;check if end of file
        je      filrea6
        or      al, al                  ;check if NUL
        je      filrea3
        cmp     al, 13                  ;check if CR
        jne     filrea2
        or      cx, cx                  ;check if more characters after CR
        jz      filrea2
        cmp     BYTE [si], 10           ;check if CR/LF combo
        je      filrea3
filrea2 cmp     dx, MAXLIN              ;check if line too long
        je      filrea7
        stosb                           ;save byte
        inc     dx                      ;adjust count
filrea3 or      cx, cx                  ;check if more bytes
        jnz     filrea1                 ;loop back if so

filrea4 pop     ds
        sub     al, al                  ;NUL
        stosb                           ;store to end of line
        mov     filptr, si              ;save pointer
        mov     filrem, cx              ;save bytes
        mov     si, linbuf              ;buffer address
        mov     cx, dx                  ;bytes returned
        or      cx, cx                  ;set ZF
        pop     di
        ret

;--- end of line

filrea5 or      dx, dx                  ;check if blank line
        jz      filrea3                 ;get next line if so
        jmps    filrea4

;--- end of file

filrea6 sub     cx, cx                  ;zero bytes
        jmps    filrea4

;--- error

filrea7 pop     ds
        pop     di
        stc
        ret
        ENDP

;========================================================================
; Write a string.
;
; In: AX= string address.
;
; Out: CY= set if error.

File_Write PROC NEAR
        push    di
        push    si
        push    es
        mov     si, ax
        mov     di, filptr              ;file pointer
        mov     es, filseg              ;buffer segment
        call    Copy_Str                ;copy string
        mov     filptr, di              ;save pointer
        pop     es
        pop     si
        pop     di
        clc
        ret
        ENDP

;========================================================================
; Load a file.
;
; In: AX= name.
;
; Out: CY= set if error.

File_Load PROC  NEAR
        push    cx
        mov     cl, OPEN_READ           ;open type
        call    FilOpn                  ;open file
        jc      filloa2
        push    ax                      ;save handle
        mov     bx, ax
        mov     cx, 0FFFFH              ;read max bytes
        sub     ax, ax                  ;buffer address
        mov     dx, filseg              ;
        call    FilRea                  ;fill buffer
        pop     bx
        jc      filloa1                 ;check if error
        sub     al, al                  ;zero error code if no error
filloa1 push    ax                      ;save error code
        mov     fillen, cx              ;save byte count
        call    FilClo                  ;close file
        call    File_Reset              ;reset pointers
        pop     ax
        sub     al, 1                   ;set carry if non-zero error code
        cmc                             ;
filloa2 pop     cx
        ret
        ENDP

;========================================================================
; Save a file.
;
; In: AX= name.
;
; Out: CY= set if error.

File_Save PROC  NEAR
        push    cx
        sub     cl, cl                  ;normal attribute
        call    FilCre                  ;open file
        jc      filsav1
        push    ax                      ;save handle
        mov     bx, ax
        sub     ax, ax                  ;buffer address
        mov     dx, filseg              ;
        mov     cx, filptr              ;bytes to write
        call    FilWri                  ;write out buffer
        pop     bx
        lahf                            ;save carry flag
        push    ax
        call    FilClo                  ;close file
        pop     ax
        jc      filsav1                 ;exit if close error
        sahf
filsav1 pop     cx
        ret
        ENDP

;========================================================================
; Display the character if non-delimiter.
;
; In: AL= character

Display_Char PROC NEAR
        cmp     al, ' '                 ;check if delimiter
        jbe     dischr1                 ;exit if so
        push    ax
        mov     ah, 2                   ;display function
        mov     dl, al                  ;character
        int     21H                     ;execute
        pop     ax
dischr1 ret
        ENDP

;========================================================================
; Display an number and a string.
;
; In: AX= number; BX= pulural string; CX= singular string.

Display_Num1 PROC NEAR
        cmp     ax, 1                   ;check if singular
        jne     disnum1
        mov     bx, cx                  ;use singular string
disnum1 push    bx
        sub     dx, dx                  ;zero high word
        mov     cx, 10                  ;base 10
        mov     bx, OFFSET numbuf       ;number buffer
        push    bx
        call    Num2Str                 ;convert to string
        dec     ax
        add     ax, OFFSET spaces
        call    MesPut                  ;display padding
        pop     ax
        call    MesPut                  ;display number
        pop     ax
        call    MesPut                  ;display string
        ret
        ENDP

;========================================================================
; Display an number and a string and a CR+LF.
;
; In: AX= number; BX= pulural string; CX= singular string.

Display_Num2 PROC NEAR
        call    Display_Num1
        mov     ax, OFFSET crlf         ;end of line
        call    MesPut                  ;display string
        ret
        ENDP

;************************************************************************
; Pop-Up Window Selection Routines

;========================================================================
; Screen area routines.

;--- load coordinates

sload   PROC    NEAR
        mov     bx, upleft              ;load upper left
        mov     cx, loright             ;load lower right
        ret
        ENDP

;--- load coordinates and call a routine

sCall   MACRO   screenr
        call    sload                   ;load coordinates
        call    screenr                 ;call routine
        ENDM

;========================================================================
; Pop up a window an allow a command to be selected.

Select  PROC    NEAR
        call    VidInit                 ;initialize video
        jnc     select1
        mov     bp, OFFSET messV
        ret

;--- dimensions

select1 call    ModDim                  ;get screen dimensions
        mov     dx, windim              ;load user size
        or      dl, dl                  ;check columns
        jnz     select2
        mov     dl, al                  ;screen columns
        shr     dl                      ;divide by two
select2 or      dh, dh                  ;check rows
        jnz     select3
        mov     dh, ah                  ;screen rows
        shr     dh                      ;divide by two
        inc     dh                      ;make it an odd number

;--- verify rows and columns

select3 cmp     dl, al                  ;check column range
        jbe     select4
        mov     dl, al
select4 cmp     dh, ah                  ;check row range
        jbe     select5
        mov     dh, ah

select5 cmp     dl, 5                   ;check column range
        jae     select6
        mov     dl, 5
select6 cmp     dh, 3                   ;check row range
        jae     select7
        mov     dh, 3

;--- location

select7 mov     cx, ax                  ;save screen dimensions
        sub     ax, dx                  ;extra rows and columns
        shr     al                      ;divide by two
        shr     ah                      ;  and get starting x - y

        mov     bx, winloc              ;load location
        or      bh, bh                  ;check if specific row
        jz      select8
        mov     ah, bh
        dec     ah
select8 or      bl, bl                  ;check if specific column
        jz      select9
        mov     al, bl
        dec     al
select9 mov     bx, ax                  ;save upper left


        add     ax, dx                  ;calculate lower right
        sub     ax, 0101H               ;

;--- check for wrap-around

        cmp     al, bl                  ;check if right greater than left
        ja      selectA
        inc     al                      ;adjustment
        sub     bl, al                  ;new right
        mov     al, 255                 ;new left

selectA cmp     ah, bh                  ;check if bottom greater than top
        ja      selectB
        inc     ah                      ;adjustment
        sub     bh, ah                  ;new top
        mov     ah, 255                 ;new bottom

;--- verify row and column

selectB sub     cx, 0101h               ;calculate max rows and columns
        cmp     al, cl                  ;check column
        jb      selectC
        sub     al, cl                  ;calculate adjustment
        sub     bl, al                  ;adjust left column
        mov     al, cl                  ;new right column

selectC cmp     ah, ch                  ;check row
        jb      selectD
        sub     ah, ch                  ;calculate adjustment
        sub     bh, ah                  ;adjust top row
        mov     ah, ch                  ;new bottom row

selectD mov     upleft, bx              ;save upper left
        mov     loright, ax             ;save lower right

;--- other coordinates

        sub     dx, 0204H               ;adjust
        mov     texdim, dx              ;save text dimensions

        mov     ah, dh                  ;rows
        shr     ah                      ;divide by two
        mov     BYTE upslots, ah        ;upper slots
        sub     dh, ah                  ;remaining slots
        dec     dh                      ;reduce for selected text
        mov     BYTE dnslots, dh        ;lower slots
        sub     al, al                  ;zero column offset
        add     ax, 0102H               ;border offset
        add     ax, upleft              ;adjust from upper left corner
        mov     homeloc, ax             ;save home location

;--- allocate memory

        sCall   ScrSiz                  ;get bytes required
        call    MemAll                  ;allocate memory
        jnc     selectE
        mov     bp, OFFSET messW
        ret

selectE mov     savseg, ax              ;save segment

;--- draw window

        push    ax                      ;save segment

        call    CurPos                  ;get cursor location
        mov     cursav1, ax             ;save it
        mov     ax, homeloc             ;home location
        call    CurMov                  ;move cursor
        sub     al, al
        call    CurAct                  ;disable cursor
        mov     cursav2, al             ;save last state

        pop     dx
        sub     ax, ax
        sCall   ScrGet                  ;save screen
        mov     al, winatr1             ;border attribute
        call    AtrSet                  ;set attribute
        sCall   ScrClr                  ;clear area
        sCall   DrwBox                  ;draw box

        mov     ax, homeloc             ;home location
        push    ax
        dec     al
        dec     al                      ;edge column
        call    CurMov                  ;move cursor
        mov     al, 204                 ;connector
        call    WrtChr                  ;display
        call    CurAdv                  ;advance cursor
        mov     al, 16                  ;triangle
        call    WrtChr                  ;display
        pop     ax
        add     al, texcols             ;skip text
        call    CurMov                  ;move cursor
        mov     al, 17                  ;triangle
        call    WrtChr                  ;display
        call    CurAdv                  ;advance cursor
        mov     al, 185                 ;connector
        call    WrtChr                  ;display

;--- process input

        call    Key_Loop                ;key loop

        or      di, di                                  ;check if selected
        jz      selectF
        sub     di, OFFSET TRAN_END - OFFSET RES_END    ;adjust for relocate
        mov     _loaded, di                             ;save entry
        or      _flags, BREAK                           ;clear any macros

;--- restore screen and release memory

selectF mov     dx, savseg              ;reload segment
        push    dx
        sub     ax, ax                  ;zero offset
        sCall   ScrPut                  ;restore screen

        or      _vid_flags, _vid_CVIS

        mov     ax, cursav1             ;saved cursor location
        call    CurMov                  ;move cursor
        mov     al, cursav2             ;saved state
        call    CurAct                  ;restore it

        pop     ax
        call    MemRel                  ;release memory

        sub     di, di                  ;no exit message (and clear CY)
        ret
        ENDP

;========================================================================
; Process keystrokes in selection window.
;
; Out: DI= entry.

        PROC    NEAR
keyloo1 sub     di, di                  ;zero pointer if no entries
keyloo2 call    Disp_Redraw             ;redraw window

keyloo3 call    KeyWai                  ;wait for key
        or      di, di                  ;check if no entries
        jz      keylooB

        cmp     ax, KEY_UP              ;UP
        jne     keyloo4
        call    Entry_Prev              ;get previous entry
        jc      keyloo3
        jmps    keyloo2

keyloo4 cmp     ax, KEY_DOWN            ;DOWN
        jne     keyloo5
        call    Entry_Next              ;next entry
        jc      keyloo3
        jmps    keyloo2

keyloo5 cmp     ax, KEY_PGUP            ;PAGE-UP
        jne     keyloo7
        mov     cl, texrows             ;number of rows
        sub     ch, ch                  ;zero high byte
keyloo6 call    Entry_Prev              ;get previous entries
        loop    keyloo6
        jmps    keyloo2

keyloo7 cmp     ax, KEY_PGDN            ;PAGE-DOWN
        jne     keyloo9
        mov     cl, texrows             ;number of rows
        sub     ch, ch                  ;zero high byte
keyloo8 call    Entry_Next              ;get next entries
        loop    keyloo8
        jmps    keyloo2

keyloo9 cmp     ax, KEY_HOME            ;HOME
        jne     keylooA
        mov     di, _hisbeg             ;start of table
        cmp     di, _hisend             ;check if same as end
        je      keyloo1
        jmps    keyloo2

keylooA cmp     ax, KEY_END             ;END
        jne     keylooB
Key_Loop LABEL  NEAR                    ;routine entry point
        mov     di, _hisend             ;end of table
        cmp     di, _hisbeg             ;check if same as start
        je      keyloo1
        call    Str_Prev                ;get last entry
        jmps    keyloo2

keylooB cmp     ax, KEY_ENTER           ;ENTER
        je      keylooD
        cmp     ax, KEY_CTL_C           ;check if break
        je      keylooC
        cmp     ax, KEY_ESC             ;ESC
        jne     keyloo3
keylooC sub     di, di                  ;zero return

keylooD ret
        ENDP

;========================================================================
; Display the selection window entries.
;
; In: DI= current history entry.

Disp_Redraw PROC NEAR
        push    si
        mov     ax, homeloc             ;home location
        call    CurMov                  ;position cursor
        mov     al, winatr2             ;load attribute
        call    Disp_Entry              ;set attribute

        mov     dl, -1                  ;decrement row
        mov     cx, upslots             ;lines to display
        mov     si, OFFSET Entry_Prev   ;function
        call    Disp_Entries            ;display upper lines

        mov     ax, homeloc             ;home location
        call    CurMov                  ;position cursor

        mov     dl, 1                   ;increment row
        mov     cx, dnslots             ;lines to display
        mov     si, OFFSET Entry_Next   ;function
        call    Disp_Entries            ;display lower lines
        pop     si
        ret
        ENDP

;------------------------------------------------------------------------
; Display all upper or lower entries.
;
; In: DI= current entry; DL= increment (1 or -1); SI= Entry_Prev or
;     Entry_Next; CX= number of rows.

Disp_Entries PROC NEAR
        push    di
        jcxz    disens3                 ;exit if none
disens1 push    cx
        push    dx
        call    si
        jnc     disens2
        sub     di, di
disens2 call    CurPos                  ;get cursor location
        pop     dx
        add     ah, dl                  ;increment/decrement
        push    dx
        call    CurMov                  ;position cursor
        mov     al, winatr3             ;load attribute
        call    Disp_Entry              ;display
        pop     dx
        pop     cx
        loop    disens1                 ;loop for entry count
disens3 pop     di
        ret
        ENDP

;------------------------------------------------------------------------
; Display a selection window entry.
;
; In: DI= entry; AL= attribute.

Disp_Entry PROC NEAR
        call    AtrSet                  ;set attribute
        call    CurPos                  ;get location
        push    ax                      ;save on stack
        mov     cl, texcols             ;columns
        or      di, di                  ;check if defined
        jz      disent2
        call    Str_Length              ;get length
        cmp     cl, texcols             ;check if too long
        jbe     disent1
        mov     cl, texcols             ;use max columns
disent1 push    cx
        mov     ax, di                  ;load string
        call    WrtStrc                 ;display
        call    CurAdv
        pop     cx
        sub     cl, texcols             ;extra columns
        jz      disent3
        neg     cl                      ;adjust
disent2 mov     al, ' '                 ;fill with spaces
        call    WrtChrs                 ;display
disent3 pop     ax                      ;restore location
        call    CurMov                  ;move cursor
        ret
        ENDP

;========================================================================
; Switch to the previous or next history entries.
;
; In: DI= current entry.
;
; Out: DI= updated entry; CY= set if could not switch.

;--- previous entry

Entry_Prev PROC NEAR
        or      di, di                  ;check if none
        jz      entpre1
        cmp     di, _hisbeg             ;check if at start
        je      entpre1                 ;exit if so
        call    Str_Prev                ;previous entry
        clc
        ret
entpre1 stc
        ret
        ENDP

;--- next entry

Entry_Next PROC NEAR
        or      di, di                  ;check if none
        jz      entnex1
        push    di
        mov     di, _hisend             ;load end
        call    Str_Prev                ;get last entry
        mov     ax, di                  ;save in AX
        pop     di
        cmp     di, ax                  ;check if at end
        je      entnex1                 ;exit if so
        call    Str_Next                ;next entry
        clc
        ret
entnex1 stc
        ret
        ENDP

;************************************************************************
; Installation Data.  Note: cannot use ORG +n for the uninitialized data
; at the end because saved macros (which are append to this file) must be
; loaded at the proper memory location.

rseg    DW      0                       ;resident segment
iflags  DB      0                       ;installation flags
sflags  DB      ?                       ;switch flags
serror  DW      0                       ;switch error characters
mactyp  DB      ?                       ;macro definition type
cmdlen  DW      0                       ;command length (in cmdbuf)
heapsiz DW      ?                       ;size of heap
heapmax DW      ?                       ;maximum size of heap

pname   DW      ?                       ;program name address
cname   DW      ?                       ;config name address
numbuf  DS      6                       ;number conversion buffer
linbuf  DW      ?                       ;line buffer address
keybuf  DW      ?                       ;key buffer address

spaces  DB      '    ', 0               ;spaces padding number
sname   DB      'TODDY', 0              ;select command
tname   DB      'TODDY.COM', 0          ;Toddy file name
ename   DB      '.CFG', 0               ;configuration extension
digits  DB      '0123456789ABCDEF'      ;number digits

;--- argument table

argtab  DW      ?                       ;table address
argtab2 DW      ?                       ;table 2 address (for Load_Config)
argptr  DW      ?                       ;pointer
arglev  DW      ?                       ;level

;--- file data

filseg  DW      0                       ;buffer segment
filptr  DW      ?                       ;buffer pointer
fillen  DW      ?                       ;read bytes
filrem  DW      ?                       ;remaining bytes

;--- select data

cursav1 DW      ?                       ;saved cursor location
cursav2 DB      ?                       ;save cursor state
savseg  DW      ?                       ;saved screen area

texdim  LABEL   WORD                    ;text dimensions
texcols DB      ?                       ;text columns
texrows DB      ?                       ;text rows

upleft  DW      ?                       ;upper left coordinates
loright DW      ?                       ;lower right coordinates
homeloc DW      ?                       ;home (selected text) location
upslots DW      0                       ;upper slots
dnslots DW      0                       ;lower slots

;--- messages

equals  DB      '=', 0
crlf    DB      13, 10, 0
error1  DB      ' (/', 0
error2  DB      ')', 0
mquote  DB      ?, 0

title   DB      '; Toddy Configuration File', 13, 10, 0

banner  DB      13,10,'Toddy  Version '
        DB      VER_HI+'0','.', VER_LO / 10 + '0', VER_LO \ 10 + '0'
        DB      '  Eric Tauck',0

mess1   DB      'Options sent to resident Toddy',0
mess2   DB      'Error in switches',0
mess3   DB      'Not enough macro space for definition',0
mess4   DB      'Installed and resident',0
mess5   DB      'Removed from memory',0
mess6   DB      'Must be resident to use switch',0
mess7   DB      'Cannot use switch when resident',0
mess8   DB      'Not enough macro space to load file',0
mess9   DB      'Cannot uninstall, interrupts hooked',0
messA   DB      'Cannot undefine, macro not defined',0
messB   DB      'Syntax error in key file',0
messC   DB      'Options written to executable',0
messD   DB      'Unable to write current settings',0
messE   DB      'Not enough memory to install',0
messF   DB      'Memory sizes are too big',0
messG   DB      'Multiplex number in use by another program',0
messH   DB      'No macros defined',0
messI   DB      'Error writing macro file',0
messJ   DB      'Error reading macro file',0
messK   DB      'Macros read from file',0
messL   DB      'Macros written to file',0
messM   DB      'Resident Toddy enabled',0
messN   DB      'Resident Toddy disabled',0
messO   DB      'Command history written to file',0
messP   DB      'Error writing command history file',0
messQ   DB      'Macro size too small',0
messR   DB      'Not enough memory',0
messS   DB      'Command history read from file',0
messT   DB      'Error reading command history file',0
messU   DB      'Syntax error in macro file',0
messV   DB      'Cannot select, invalid video mode',0
messW   DB      'Cannot select, not enough memory',0
messX   DB      'Cannot forget, key not defined',0
messY   DB      'Cannot forget, macro not defined',0
messZ   DB      'Invalid switch in configuration file',0
messAA  DB      'Cannot undefine, key not defined',0
messBB  DB      'Keys written to file',0
messCC  DB      'Error writing key file',0
messDD  DB      'No keys defined',0
messEE  DB      'Keys read from file',0
messFF  DB      'Error reading key file',0
messGG  DB      'Configuration read from file',0
messHH  DB      'Configuration written to file',0

alloc1  DB      ' bytes allocated for history',0
alloc2  DB      ' bytes allocated for macros',0
alloc3  DB      ' bytes allocated for names',13,10,0
alloc4  DB      ' bytes allocated total',0
alloc5  DB      ' bytes used by macros',0
alloc6  DB      ' bytes available',13,10,0
alloc7  DB      ' macro levels (',0
alloc8  DB      ' bytes)',0
alloc9  DB      ' byte keyboard buffer',0
allocA  DB      ' byte input buffer',13,10,0
allocB  DB      ' bytes of resident data',0
allocC  DB      ' macro level (',0
allocD  DB      ' byte allocated for history',0
allocE  DB      ' byte allocated for macros',0
allocF  DB      ' byte available',13,10,0
allocG  DB      ' byte variable data area',0

;--- switch table flags

QUI     EQU     80H           ;quit after switch (no install)
RES     EQU     40H           ;must be resident before switch
NON     EQU     20H           ;must be non-resident for switch
BAN     EQU     10H           ;show banner
HID     EQU     08H           ;hide banner
CFG     EQU     04H           ;not allowed in configuration file

; The low 3 bits are the switch priority:
;
;   0 verbose setting
;   1 uninstall
;   2 load configuration
;   3 normal
;   4 read keys/macros
;   5 write keys/macros
;   6 save configuration
;   7 display status

;--- switch table

switches LABEL  BYTE
  DB 'U ', OFFSET Option22, 1, 0 OR CFG OR QUI OR RES   ;uninstall
  DB 'E ', OFFSET Option20, 3, 0                        ;enable
  DB 'D ', OFFSET Option21, 3, 0                        ;disable
  DB 'R ', OFFSET Option57, 3, 1                        ;run command
  DB 'W ', OFFSET Option23, 6, 0 OR CFG                 ;write settings
  DB 'OR', OFFSET Option88, 2, 0 OR CFG                 ;read configuration
  DB 'OW', OFFSET Option89, 6, 0 OR CFG                 ;write configuration
  DB 'A ', OFFSET Option49, 7, 0 OR CFG OR QUI OR BAN   ;display allocations
  DB '? ', OFFSET Option27, 0, 0 OR CFG OR QUI OR HID   ;display help

  DB 'Q ', OFFSET Option39, 0, 0                        ;quiet mode
  DB 'V ', OFFSET Option40, 0, 0                        ;verbose mode
  DB 'B ', OFFSET Option55, 3, 0 OR NON                 ;buffer size
  DB 'M ', OFFSET Option32, 3, 0                        ;multiplex
  DB 'BS', OFFSET Option34, 3, 0                        ;allowed size

  DB 'US', OFFSET Option47, 3, 0                        ;uppercase signif
  DB 'UI', OFFSET Option48, 3, 0                        ;uppercase ignore
  DB 'WU', OFFSET Option86, 3, 0                        ;public Windows data
  DB 'WP', OFFSET Option87, 3, 0                        ;private Windows data

  DB 'IP', OFFSET Option24, 3, 0                        ;preserve insert
  DB 'IR', OFFSET Option10, 3, 0                        ;reset insert
  DB 'II', OFFSET Option08, 3, 0                        ;insert mode
  DB 'IO', OFFSET Option09, 3, 0                        ;overwrite mode

  DB 'CB', OFFSET Option80, 3, 0                        ;beginning of key
  DB 'CE', OFFSET Option81, 3, 0                        ;end of key command
  DB 'CL', OFFSET Option05, 3, 0                        ;literal character
  DB 'CP', OFFSET Option06, 3, 0                        ;parameter character
  DB 'CC', OFFSET Option07, 3, 0                        ;chain character
  DB 'CQ', OFFSET Option50, 3, 0                        ;quote character
  DB 'CX', OFFSET Option51, 3, 0                        ;text quote character one
  DB 'CY', OFFSET Option52, 3, 0                        ;text quote character two
  DB 'CM', OFFSET Option56, 3, 0                        ;multiple arg char
  DB 'CV', OFFSET Option73, 3, 0                        ;start variable char
  DB 'CW', OFFSET Option74, 3, 0                        ;end variable char

  DB 'AT', OFFSET Option11, 3, 0                        ;text attribute
  DB 'AC', OFFSET Option12, 3, 0                        ;cursor attribute
  DB 'AE', OFFSET Option13, 3, 0                        ;end attribute
  DB 'AB', OFFSET Option61, 3, 0                        ;select border attr
  DB 'AH', OFFSET Option62, 3, 0                        ;select high attr
  DB 'AX', OFFSET Option63, 3, 0                        ;select text attr

  DB 'TE', OFFSET Option16, 3, 0                        ;define entry
  DB 'TI', OFFSET Option17, 3, 0                        ;define insert
  DB 'TO', OFFSET Option18, 3, 0                        ;define overwrite
  DB 'TX', OFFSET Option19, 3, 0                        ;define exit
  DB 'TC', OFFSET Option14, 3, 0                        ;custom cursor
  DB 'TS', OFFSET Option15, 3, 0                        ;system cursor

  DB 'HS', OFFSET Option03, 3, 0 OR NON                 ;history size
  DB 'HM', OFFSET Option01, 3, 0                        ;minimum length
  DB 'HP', OFFSET Option90, 3, 0                        ;pure history 
  DB 'HI', OFFSET Option91, 3, 0                        ;impure history
  DB 'HD', OFFSET Option66, 3, 0                        ;history delete
  DB 'HK', OFFSET Option67, 3, 0                        ;history keep
  DB 'HU', OFFSET Option02, 3, 0                        ;save unique input
  DB 'HA', OFFSET Option25, 3, 0                        ;save all input
  DB 'HT', OFFSET Option64, 3, 0                        ;trace recover
  DB 'HN', OFFSET Option65, 3, 0                        ;no trace recover
  DB 'HL', OFFSET Option44, 3, 0                        ;history lock
  DB 'HF', OFFSET Option45, 3, 0                        ;history free
  DB 'HC', OFFSET Option43, 3, 0                        ;history clear
  DB 'HR', OFFSET Option42, 4, 0 OR CFG                 ;history read
  DB 'HW', OFFSET Option41, 5, 0 OR CFG OR RES          ;history write

  DB '  ', OFFSET Option36, 3, 3                        ;define macro

  DB 'MS', OFFSET Option04, 3, 0 OR NON                 ;macro size
  DB 'MN', OFFSET Option29, 3, 0 OR NON                 ;macro nest
  DB 'MA', OFFSET Option96, 3, 0                        ;append extra args
  DB 'MI', OFFSET Option95, 3, 0                        ;ignore extra args
  DB 'MD', OFFSET Option26, 3, 2                        ;define macro
  DB 'ME', OFFSET Option33, 3, 2                        ;exchange macro
  DB 'MU', OFFSET Option28, 3, 1                        ;undefine macro
  DB 'MF', OFFSET Option31, 3, 1                        ;forget macro
  DB 'MC', OFFSET Option30, 3, 0                        ;clear all macros
  DB 'MR', OFFSET Option37, 4, 0 OR CFG                 ;read macros
  DB 'MW', OFFSET Option38, 5, 0 OR CFG                 ;write macros
  DB 'ML', OFFSET Option35, 7, 0 OR CFG OR QUI OR BAN   ;list all macros

  DB 'KS', OFFSET Option68, 3, 0 OR NON                 ;key buffer size
  DB 'KD', OFFSET Option76, 3, 2                        ;define key
  DB 'KE', OFFSET Option77, 3, 2                        ;exchange key
  DB 'KU', OFFSET Option78, 3, 1                        ;undefine key
  DB 'KF', OFFSET Option79, 3, 1                        ;forget key
  DB 'KC', OFFSET Option69, 3, 0                        ;clear keys
  DB 'KR', OFFSET Option75, 4, 0 OR CFG                 ;read keys
  DB 'KW', OFFSET Option72, 5, 0 OR CFG                 ;write keys
  DB 'KL', OFFSET Option71, 7, 0 OR CFG OR QUI OR BAN   ;list keys
  DB 'KT', OFFSET Option92, 3, 0                        ;key input type

  DB 'NS', OFFSET Option70, 3, 0 OR NON                 ;name size
  DB 'NT', OFFSET Option46, 3, 0                        ;name file types
  DB 'NL', OFFSET Option53, 3, 0                        ;lowercase names
  DB 'NU', OFFSET Option54, 3, 0                        ;uppercase names
  DB 'NC', OFFSET Option82, 3, 0                        ;clean dir names
  DB 'NB', OFFSET Option83, 3, 0                        ;backslashed dir

  DB 'VN', OFFSET Option84, 3, 0 OR NON                 ;variable nest
  DB 'VL', OFFSET Option85, 3, 0 OR NON                 ;variable length

  DB 'SC', OFFSET Option59, 3, 0                        ;select columns
  DB 'SR', OFFSET Option60, 3, 0                        ;select rows
  DB 'SX', OFFSET Option93, 3, 0                        ;select column
  DB 'SY', OFFSET Option94, 3, 0                        ;select row
  DB 'S ', OFFSET Option58, 0, 0 OR CFG OR RES OR QUI OR HID ;select window
  DB 0

;--- help messages

prompt1 LABEL BYTE
 DB '-- press esc to quit or any other key to continue --'
 DB 0

prompt2 LABEL BYTE
 DB '-- press p for other page or any other key to quit --'
 DB 0

prompt3 LABEL BYTE
 DB 13, '                                                     ',13
 DB 0

helpt LABEL BYTE
 DB 13,10
 DB 'Toddy Version '
 DB VER_HI+'0','.', VER_LO / 10 + '0', VER_LO \ 10 + '0'
 DB '              Eric Tauck',13,10
 DB 'DOS Command Line Editor         1304 Deerpass Road',13,10
 DB '                                Marengo, IL 60152-9644',13,10
 DB 'warp@earthling.net              U.S.A.',13,10
 DB 13,10
 DB 'Usage: TODDY [switches] [macro=text]   ddd/xxx/',39,'s',39
 DB '/file = dec/hex/string/file',13,10
 DB 0

help1 LABEL BYTE
 DB '/A       show allocations  /CVddd   var start char    /HRfile  read history',13,10
 DB '/ABxx    win border attr   /CWddd   var end char      /HSddddd history size',13,10
 DB '/ACxx    cursor attr       /CXddd   text quote char   /HT      trace recover',13,10
 DB '/AExx    end attribute     /CYddd   text quote char   /HU      save unique',13,10
 DB '/AHxx    win high attr     /D       disable Toddy     /HWfile  write history',13,10
 DB '/ATxx    text attribute    /E       enable Toddy      /II      insert mode',13,10
 DB '/AXxx    win text attr     /HA      save all input    /IO      overwrite mode',13,10
 DB '/Bddddd  buffer size       /HC      clear history     /IP      preserve insert',13,10
 DB '/BSddd   restrict buffer   /HD      delete selected   /IR      reset insert',13,10
 DB '/CBddd   begin key char    /HF      free history      /KC      clear keys',13,10
 DB '/CCddd   chain char        /HI      impure history    /KD',39,'s',39,'s',39,' define key',13,10
 DB '/CEddd   end key char      /HK      keep selected     /KE',39,'s',39,'s',39,' exchange key',13,10
 DB '/CLddd   literal char      /HL      lock history      /KF',39,'s',39,'   forget key',13,10
 DB '/CMddd   multiple char     /HP      pure history      /KL      list keys',13,10
 DB '/CPddd   parameter char    /HMddd   min save length   /KRfile  read keys',13,10
 DB '/CQddd   quote character   /HN      no trace recover  /KSddddd key size'
 DB 0

help2 LABEL BYTE
 DB '/KTd     key type: 7,8     /NB      backslash dirs    /TExxxx  entry cursor',13,10
 DB '/KU',39,'s',39,'   undefine key      /NC      clean dirs        /TIxxxx  insert cursor',13,10
 DB '/KWfile  write keys        /NL      lowercase names   /TOxxxx  overwrite cursor',13,10
 DB '/Mddd    multiplex number  /NSddddd name size         /TS      system cursor',13,10
 DB '/MA      append extr args  /NTxx    name file types   /TXxxxx  exit cursor',13,10
 DB '/MC      clear all macros  /NU      uppercase names   /U       uninstall Toddy',13,10
 DB '/MD',39,'s',39,'s',39,' define macro      /ORfile  read config file  /UI      uppercase ignore',13,10
 DB '/ME',39,'s',39,'s',39,' exchange macro    /OWfile  write config      /US      uppercase signif',13,10
 DB '/MF',39,'s',39,'   forget macro      /Q       quiet mode        /V       verbose mode',13,10
 DB '/MI      ignore extr args  /R',39,'s',39,'    run a command     /VLddd   max var length',13,10
 DB '/ML      list macros       /S       open select win   /VNddd   max var nest',13,10
 DB '/MNddd   max macro nest    /SCddd   window columns    /W       write switches',13,10
 DB '/MRfile  read macros       /SRddd   window rows       /WP      private Win data',13,10
 DB '/MSddddd macro size        /SXddd   window column     /WU      public Win data',13,10
 DB '/MU',39,'s',39,'   undefine macro    /SYddd   window row',13,10
 DB '/MWfile  write macros      /TC      custom cursor'
 DB 0

;--- code strings

cmdstr  LABEL   BYTE
        DB      CMD_BKSP,       'Backspace',0
        DB      CMD_TAB,        'Tab',0
        DB      CMD_ENTER,      'Enter',0
        DB      CMD_ESC,        'Escape',0
        DB      CMD_SHF_TAB,    'Shift-Tab',0
        DB      CMD_CTL_ENTER,  'Ctrl-Enter',0
        DB      CMD_CTL_SLASH,  'Ctrl-Slash',0
        DB      CMD_CTL_MINUS,  'Ctrl-Minus',0
        DB      CMD_CTL_BKSP,   'Ctrl-Backspace',0
        DB      CMD_ALT_MINUS,  'Alt-Minus',0
        DB      CMD_ALT_EQUAL,  'Alt-Equal',0
        DB      CMD_CTL_2,      'Ctrl-2',0
        DB      CMD_CTL_6,      'Ctrl-6',0
        DB      CMD_ALT_0,      'Alt-0',0
        DB      CMD_ALT_1,      'Alt-1',0
        DB      CMD_ALT_2,      'Alt-2',0
        DB      CMD_ALT_3,      'Alt-3',0
        DB      CMD_ALT_4,      'Alt-4',0
        DB      CMD_ALT_5,      'Alt-5',0
        DB      CMD_ALT_6,      'Alt-6',0
        DB      CMD_ALT_7,      'Alt-7',0
        DB      CMD_ALT_8,      'Alt-8',0
        DB      CMD_ALT_9,      'Alt-9',0
        DB      CMD_CTL_A,      'Ctrl-A',0
        DB      CMD_CTL_B,      'Ctrl-B',0
        DB      CMD_CTL_C,      'Ctrl-C',0
        DB      CMD_CTL_D,      'Ctrl-D',0
        DB      CMD_CTL_E,      'Ctrl-E',0
        DB      CMD_CTL_F,      'Ctrl-F',0
        DB      CMD_CTL_G,      'Ctrl-G',0
        DB      CMD_CTL_H,      'Ctrl-H',0
        DB      CMD_CTL_I,      'Ctrl-I',0
        DB      CMD_CTL_J,      'Ctrl-J',0
        DB      CMD_CTL_K,      'Ctrl-K',0
        DB      CMD_CTL_L,      'Ctrl-L',0
        DB      CMD_CTL_M,      'Ctrl-M',0
        DB      CMD_CTL_N,      'Ctrl-N',0
        DB      CMD_CTL_O,      'Ctrl-O',0
        DB      CMD_CTL_P,      'Ctrl-P',0
        DB      CMD_CTL_Q,      'Ctrl-Q',0
        DB      CMD_CTL_R,      'Ctrl-R',0
        DB      CMD_CTL_S,      'Ctrl-S',0
        DB      CMD_CTL_T,      'Ctrl-T',0
        DB      CMD_CTL_U,      'Ctrl-U',0
        DB      CMD_CTL_V,      'Ctrl-V',0
        DB      CMD_CTL_W,      'Ctrl-W',0
        DB      CMD_CTL_X,      'Ctrl-X',0
        DB      CMD_CTL_Y,      'Ctrl-Y',0
        DB      CMD_CTL_Z,      'Ctrl-Z',0
        DB      CMD_ALT_A,      'Alt-A',0
        DB      CMD_ALT_B,      'Alt-B',0
        DB      CMD_ALT_C,      'Alt-C',0
        DB      CMD_ALT_D,      'Alt-D',0
        DB      CMD_ALT_E,      'Alt-E',0
        DB      CMD_ALT_F,      'Alt-F',0
        DB      CMD_ALT_G,      'Alt-G',0
        DB      CMD_ALT_H,      'Alt-H',0
        DB      CMD_ALT_I,      'Alt-I',0
        DB      CMD_ALT_J,      'Alt-J',0
        DB      CMD_ALT_K,      'Alt-K',0
        DB      CMD_ALT_L,      'Alt-L',0
        DB      CMD_ALT_M,      'Alt-M',0
        DB      CMD_ALT_N,      'Alt-N',0
        DB      CMD_ALT_O,      'Alt-O',0
        DB      CMD_ALT_P,      'Alt-P',0
        DB      CMD_ALT_Q,      'Alt-Q',0
        DB      CMD_ALT_R,      'Alt-R',0
        DB      CMD_ALT_S,      'Alt-S',0
        DB      CMD_ALT_T,      'Alt-T',0
        DB      CMD_ALT_U,      'Alt-U',0
        DB      CMD_ALT_V,      'Alt-V',0
        DB      CMD_ALT_W,      'Alt-W',0
        DB      CMD_ALT_X,      'Alt-X',0
        DB      CMD_ALT_Y,      'Alt-Y',0
        DB      CMD_ALT_Z,      'Alt-Z',0
        DB      CMD_LEFT,       'Left',0
        DB      CMD_RIGHT,      'Right',0
        DB      CMD_UP,         'Up',0
        DB      CMD_DOWN,       'Down',0
        DB      CMD_PGUP,       'PageUp',0
        DB      CMD_PGDN,       'PageDown',0
        DB      CMD_HOME,       'Home',0
        DB      CMD_END,        'End',0
        DB      CMD_CTL_LEFT,   'Ctrl-Left',0
        DB      CMD_CTL_RIGHT,  'Ctrl-Right',0
        DB      CMD_CTL_PGUP,   'Ctrl-PageUp',0
        DB      CMD_CTL_PGDN,   'Ctrl-PageDown',0
        DB      CMD_CTL_HOME,   'Ctrl-Home',0
        DB      CMD_CTL_END,    'Ctrl-End',0
        DB      CMD_INS,        'Insert',0
        DB      CMD_DEL,        'Delete',0
        DB      CMD_F1,         'F1',0
        DB      CMD_F2,         'F2',0
        DB      CMD_F3,         'F3',0
        DB      CMD_F4,         'F4',0
        DB      CMD_F5,         'F5',0
        DB      CMD_F6,         'F6',0
        DB      CMD_F7,         'F7',0
        DB      CMD_F8,         'F8',0
        DB      CMD_F9,         'F9',0
        DB      CMD_F10,        'F10',0
        DB      CMD_F11,        'F11',0
        DB      CMD_F12,        'F12',0
        DB      CMD_SHF_F1,     'Shift-F1',0
        DB      CMD_SHF_F2,     'Shift-F2',0
        DB      CMD_SHF_F3,     'Shift-F3',0
        DB      CMD_SHF_F4,     'Shift-F4',0
        DB      CMD_SHF_F5,     'Shift-F5',0
        DB      CMD_SHF_F6,     'Shift-F6',0
        DB      CMD_SHF_F7,     'Shift-F7',0
        DB      CMD_SHF_F8,     'Shift-F8',0
        DB      CMD_SHF_F9,     'Shift-F9',0
        DB      CMD_SHF_F10,    'Shift-F10',0
        DB      CMD_SHF_F11,    'Shift-F11',0
        DB      CMD_SHF_F12,    'Shift-F12',0
        DB      CMD_CTL_F1,     'Ctrl-F1',0
        DB      CMD_CTL_F2,     'Ctrl-F2',0
        DB      CMD_CTL_F3,     'Ctrl-F3',0
        DB      CMD_CTL_F4,     'Ctrl-F4',0
        DB      CMD_CTL_F5,     'Ctrl-F5',0
        DB      CMD_CTL_F6,     'Ctrl-F6',0
        DB      CMD_CTL_F7,     'Ctrl-F7',0
        DB      CMD_CTL_F8,     'Ctrl-F8',0
        DB      CMD_CTL_F9,     'Ctrl-F9',0
        DB      CMD_CTL_F10,    'Ctrl-F10',0
        DB      CMD_CTL_F11,    'Ctrl-F11',0
        DB      CMD_CTL_F12,    'Ctrl-F12',0
        DB      CMD_ALT_F1,     'Alt-F1',0
        DB      CMD_ALT_F2,     'Alt-F2',0
        DB      CMD_ALT_F3,     'Alt-F3',0
        DB      CMD_ALT_F4,     'Alt-F4',0
        DB      CMD_ALT_F5,     'Alt-F5',0
        DB      CMD_ALT_F6,     'Alt-F6',0
        DB      CMD_ALT_F7,     'Alt-F7',0
        DB      CMD_ALT_F8,     'Alt-F8',0
        DB      CMD_ALT_F9,     'Alt-F9',0
        DB      CMD_ALT_F10,    'Alt-F10',0
        DB      CMD_ALT_F11,    'Alt-F11',0
        DB      CMD_ALT_F12,    'Alt-F12',0
        DB      CMD_CUR_LEFT,   'CursorLeft',0
        DB      CMD_CUR_RIGHT,  'CursorRight',0
        DB      CMD_CUR_NEXT,   'CursorNext',0
        DB      CMD_CUR_PREV,   'CursorPrevious',0
        DB      CMD_CUR_HOME,   'CursorHome',0
        DB      CMD_CUR_END,    'CursorEnd',0
        DB      CMD_DEL_CHAR,   'DeleteCharacter',0
        DB      CMD_DEL_BACK,   'DeleteBackward',0
        DB      CMD_DEL_NEXT,   'DeleteNext',0
        DB      CMD_DEL_PREV,   'DeletePrevious',0
        DB      CMD_DEL_FRONT,  'DeleteHome',0
        DB      CMD_DEL_REST,   'DeleteEnd',0
        DB      CMD_DEL_LINE,   'DeleteLine',0
        DB      CMD_INS_SET,    'InsertOn',0
        DB      CMD_INS_CLEAR,  'InsertOff',0
        DB      CMD_INS_TOGGLE, 'InsertToggle',0
        DB      CMD_HIS_OFF,    'HistorySkip',0
        DB      CMD_HIS_ON,     'HistoryReset',0
        DB      CMD_HIS_ADD,    'HistoryAdd',0
        DB      CMD_HIS_DELETE, 'HistoryDelete',0
        DB      CMD_HIS_CLEAR,  'HistoryClear',0
        DB      CMD_HIS_OLDER,  'HistoryOlder',0
        DB      CMD_HIS_NEWER,  'HistoryNewer',0
        DB      CMD_HIS_TRACE,  'HistoryTrace',0
        DB      CMD_HIS_OLDEST, 'HistoryOldest',0
        DB      CMD_HIS_NEWEST, 'HistoryNewest',0
        DB      CMD_HIS_FORWARD,'HistoryForward',0
        DB      CMD_HIS_REVERSE,'HistoryBackward',0
        DB      CMD_NAM_MATCH,  'NameForward',0
        DB      CMD_NAM_PREV,   'NameBackward',0
        DB      CMD_NAM_APPEND, 'NameAppend',0
        DB      CMD_TMP_CHAR,   'TemplateCharacter',0
        DB      CMD_TMP_REM,    'TemplateAppend',0
        DB      CMD_TMP_INS,    'TemplateInsert',0
        DB      CMD_TMP_SKIP,   'TemplateSkip',0
        DB      CMD_TMP_EOF,    'TemplateEOF',0
        DB      CMD_RUN,        'Run',0
        DB      CMD_RUN_DIRECT, 'RunDirect',0
        DB      CMD_MASK,       'Mask',0
cmderr  DB      CMD_ERROR,      'ERROR',0
        DB      0

;=== start of write data

WRITE_BEG

flags5  DB      0                       ;installation flags
mtail   DW      MACDEFS                 ;bytes in macro tail
multi   DW      MPLX1 * 256 + MPLX2     ;Toddy muliplex number

field1  DB      '['                     ;start of macro command field
field2  DB      ']'                     ;end of macro command field

;=== start of relocated data; this data is copied to REL_TAR

REL_BEG

;--- miscellaneous data

_flags  DB      IRESET OR IVALUE OR INSERT OR ENABLE ;flags
        DB      0                       ;attached to flags2, see NOTE_A's
_flags2 DB      HTRACE OR DELHIS        ;more flags           ;-- linked
_flags4 DB      SLASHED OR PRIVATE      ;more flags (flags4)  ;
_minlen DB      0                       ;minimum length
_liter  DB      0                       ;literal field
_quote1 DB      '"'                     ;quote character
_quote2 DB      0                       ;other quote character
_allarg DB      '*'                     ;argument continuation character
_atext  DB      07H                     ;text color
_aend   DB      07H                     ;end color
_acurs  DB      07H                     ;cursor color
_cins   DW      DCURSOR                 ;insert scan pattern
_cover  DW      DCURSOR                 ;overwrite scan pattern
_cexit  DW      DCURSOR                 ;exit scan pattern
_sizchk DB      0                       ;buffer size check (0 if no check)
_namsiz DW      256                     ;size of file name save area
_namtyp DW      0000H                   ;attribute for file name completion
_namcas DW      OFFSET To_Upper         ;name case routine
_keyfun DB      8                       ;key input function

_inpsiz DW      512                     ;input buffer size
_keysiz DW      128                     ;key code buffer size
_macnes DW      10                      ;macro nest
_varmax DB      10                      ;variable nest
_varlen DW      16                      ;max variable length

heap_h  DW      256                     ;default history size
heap_m  DW      512                     ;default macro size

;--- substitution table

_param  DB      '$',  '$'               ;parameter character
_chain  DB      20,   'T'               ;chain character
_remove DB      '`',  'Q'               ;removable quote character
        DB      '<',  'L'
        DB      '>',  'G'
        DB      '|',  'B'

_varchr1 DB     '('                     ;start of variable
_varchr2 DB     ')'                     ;end of variable

;--- select data

winatr1 DB      07h                     ;border attribute
winatr2 DB      70h                     ;selected attribute
winatr3 DB      07h                     ;text attribute

windim  LABEL   WORD                    ;window dimensions (incl border)
wincols DB      0                       ;window columns
winrows DB      0                       ;window rows

winloc  LABEL   WORD                    ;window location
wincol  DB      0                       ;window column
winrow  DB      0                       ;window row

WRITE_SIZE EQU  $ - OFFSET WRITE_BEG    ;bytes of configuration data

_curhis DW      0                       ;current history entry
_cdef   DW      ?                       ;default cursor
_loaded DW      0                       ;preloaded history (reset on entry)

;--- heap addresses

_hisbeg DW      ?                       ;beginning of history
_hisend DW      ?                       ;end of history
_macbeg DW      ?                       ;beginning of macros
_macend DW      ?                       ;end of macros

REL_SIZE EQU    $ - OFFSET REL_BEG      ;end of relocation data

        DB      0                       ;NUL for start of local heap

TRAN_END                                ;end of transient code and data

macdef  LABEL   BYTE
        DB      CMD_CTL_V, COMMAND, 0,      CMD_MASK, COMMAND, 0
        DB      CMD_CTL_R, COMMAND, 0,      CMD_DEL_NEXT, COMMAND, 0
        DB      CMD_CTL_L, COMMAND, 0,      CMD_DEL_PREV, COMMAND, 0
        DB      CMD_CTL_K, COMMAND, 0,      CMD_DEL_LINE, COMMAND
        DB                                  CMD_HIS_ADD, COMMAND, 0
        DB      CMD_CTL_D, COMMAND, 0,      CMD_HIS_DELETE, COMMAND, 0
        DB      CMD_F10, COMMAND, 0,        CMD_NAM_APPEND, COMMAND, 0
        DB      CMD_SHF_F9, COMMAND, 0,     CMD_NAM_PREV, COMMAND, 0
        DB      CMD_F9, COMMAND, 0,         CMD_NAM_MATCH, COMMAND, 0
        DB      CMD_ALT_F7, COMMAND, 0,     CMD_HIS_CLEAR, COMMAND, 0
        DB      CMD_F6, COMMAND, 0,         CMD_TMP_EOF, COMMAND, 0
        DB      CMD_F5, COMMAND, 0,         CMD_DEL_LINE, COMMAND
        DB                                  CMD_HIS_ADD, COMMAND, 0
        DB      CMD_F4, COMMAND, 0,         CMD_TMP_SKIP, COMMAND, 0
        DB      CMD_F3, COMMAND, 0,         CMD_TMP_REM, COMMAND, 0
        DB      CMD_F2, COMMAND, 0,         CMD_TMP_INS, COMMAND, 0
        DB      CMD_F1, COMMAND, 0,         CMD_TMP_CHAR, COMMAND, 0
        DB      CMD_CTL_ENTER, COMMAND, 0,  CMD_RUN_DIRECT, COMMAND
        DB                                  CMD_HIS_OFF, COMMAND, 0
        DB      CMD_ENTER, COMMAND, 0,      CMD_RUN, COMMAND, 0
        DB      CMD_DEL, COMMAND, 0,        CMD_DEL_CHAR, COMMAND, 0
        DB      CMD_ESC, COMMAND, 0,        CMD_DEL_LINE, COMMAND, 0
        DB      CMD_INS, COMMAND, 0,        CMD_INS_TOGGLE, COMMAND, 0
        DB      CMD_CTL_BKSP, COMMAND, 0,   CMD_DEL_PREV, COMMAND, 0
        DB      CMD_BKSP, COMMAND, 0,       CMD_DEL_BACK, COMMAND, 0
        DB      CMD_SHF_TAB, COMMAND, 0,    CMD_HIS_FORWARD, COMMAND, 0
        DB      CMD_TAB, COMMAND, 0,        CMD_HIS_REVERSE, COMMAND, 0
        DB      CMD_CTL_PGDN, COMMAND, 0,   CMD_HIS_NEWEST, COMMAND, 0
        DB      CMD_CTL_PGUP, COMMAND, 0,   CMD_HIS_OLDEST, COMMAND, 0
        DB      CMD_CTL_END, COMMAND, 0,    CMD_DEL_REST, COMMAND, 0
        DB      CMD_CTL_HOME, COMMAND, 0,   CMD_DEL_FRONT, COMMAND, 0
        DB      CMD_CTL_RIGHT, COMMAND, 0,  CMD_CUR_NEXT, COMMAND, 0
        DB      CMD_CTL_LEFT, COMMAND, 0,   CMD_CUR_PREV, COMMAND, 0
        DB      CMD_PGDN, COMMAND, 0,       CMD_HIS_NEWEST, COMMAND, 0
        DB      CMD_PGUP, COMMAND, 0,       CMD_HIS_OLDEST, COMMAND, 0
        DB      CMD_END, COMMAND, 0,        CMD_CUR_END, COMMAND, 0
        DB      CMD_HOME, COMMAND, 0,       CMD_CUR_HOME, COMMAND, 0
        DB      CMD_DOWN, COMMAND, 0,       CMD_HIS_TRACE, COMMAND, 0
        DB      CMD_UP, COMMAND, 0,         CMD_HIS_OLDER, COMMAND, 0
        DB      CMD_RIGHT, COMMAND, 0,      CMD_TMP_CHAR, COMMAND, 0
        DB      CMD_LEFT, COMMAND, 0,       CMD_CUR_LEFT, COMMAND, 0
MACDEFS EQU     $ - OFFSET macdef
